/*
 * Decompiled with CFR 0.152.
 */
package tech.aroma.data.cassandra;

import com.datastax.driver.core.BatchStatement;
import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.querybuilder.Delete;
import com.datastax.driver.core.querybuilder.QueryBuilder;
import java.time.Instant;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import javax.inject.Inject;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sir.wellington.alchemy.collections.lists.Lists;
import sir.wellington.alchemy.collections.sets.Sets;
import tech.aroma.data.UserRepository;
import tech.aroma.data.assertions.RequestAssertions;
import tech.aroma.thrift.LengthOfTime;
import tech.aroma.thrift.TimeUnit;
import tech.aroma.thrift.User;
import tech.aroma.thrift.exceptions.InvalidArgumentException;
import tech.aroma.thrift.exceptions.OperationFailedException;
import tech.aroma.thrift.exceptions.UserDoesNotExistException;
import tech.aroma.thrift.functions.TimeFunctions;
import tech.sirwellington.alchemy.arguments.Arguments;
import tech.sirwellington.alchemy.arguments.assertions.Assertions;
import tech.sirwellington.alchemy.arguments.assertions.StringAssertions;

final class CassandraUserRepository
implements UserRepository {
    private static final Logger LOG = LoggerFactory.getLogger(CassandraUserRepository.class);
    private final Session cassandra;
    private final QueryBuilder queryBuilder;
    private final Function<Row, User> userMapper;

    @Inject
    CassandraUserRepository(Session cassandra, QueryBuilder queryBuilder, Function<Row, User> userMapper) {
        Arguments.checkThat((Object)cassandra, (Object[])new Object[]{queryBuilder, userMapper}).are(Assertions.notNull());
        this.cassandra = cassandra;
        this.queryBuilder = queryBuilder;
        this.userMapper = userMapper;
    }

    @Override
    public void saveUser(User user) throws TException {
        Arguments.checkThat((Object)user).throwing(InvalidArgumentException.class).is(RequestAssertions.validUser());
        Statement statement = this.createStatementToSaveUser(user);
        LOG.debug("Executing statement in Cassandra to save user {}", (Object)user);
        this.tryToExecute(statement);
    }

    @Override
    public User getUser(String userId) throws TException {
        this.checkUserId(userId);
        LOG.debug("Executing query to get user with ID {}", (Object)userId);
        Statement query = this.createQueryToGetUser(userId);
        ResultSet results = this.tryToExecute(query);
        Row row = results.one();
        Arguments.checkThat((Object)row).throwing(UserDoesNotExistException.class).usingMessage("Could not find user with ID: " + userId).is(Assertions.notNull());
        User user = this.convertRowToUser(row);
        return user;
    }

    @Override
    public void deleteUser(String userId) throws TException {
        this.checkUserId(userId);
        User user = this.getUser(userId);
        Statement deleteUserStatement = this.createQueryToDeleteUser(user);
        LOG.debug("Executing Statement to delete user with ID {}", (Object)userId);
        this.tryToExecute(deleteUserStatement);
    }

    @Override
    public boolean containsUser(String userId) throws TException {
        this.checkUserId(userId);
        Statement selectStatement = this.createQueryToCheckExistenceFor(userId);
        LOG.info("Issuing query to check if user with ID [{}] exists", (Object)userId);
        ResultSet results = this.tryToExecute(selectStatement);
        long result = results.one().getLong(0);
        return result > 0L;
    }

    @Override
    public User getUserByEmail(String emailAddress) throws TException {
        Arguments.checkThat((Object)emailAddress).throwing(InvalidArgumentException.class).usingMessage("email cannot be empty").is(StringAssertions.nonEmptyString());
        Statement query = this.createQueryToGetUserByEmail(emailAddress);
        LOG.debug("Issuing CQL Query to find user with email {}", (Object)emailAddress);
        ResultSet results = this.tryToExecute(query);
        Row row = results.one();
        Arguments.checkThat((Object)row).throwing(UserDoesNotExistException.class).usingMessage("Could not find user with email: " + emailAddress).is(Assertions.notNull());
        User user = this.convertRowToUser(row);
        return user;
    }

    @Override
    public User findByGithubProfile(String githubProfile) throws TException {
        ResultSet results;
        Arguments.checkThat((Object)githubProfile).throwing(InvalidArgumentException.class).usingMessage("github profile cannot be empty").is(StringAssertions.nonEmptyString());
        Statement query = this.createQueryToGetUsersByGithubProfile(githubProfile);
        try {
            results = this.cassandra.execute(query);
        }
        catch (Exception ex) {
            LOG.error("Failed to query Cassandra Table {} for profile {}", new Object[]{"Users_By_Github_Profile", githubProfile, ex});
            throw new OperationFailedException("Could not query for profile: " + ex.getMessage());
        }
        Row row = results.one();
        Arguments.checkThat((Object)row).throwing(UserDoesNotExistException.class).usingMessage("Could not find user with Github Profile: " + githubProfile).is(Assertions.notNull());
        User user = this.convertRowToUser(row);
        return user;
    }

    @Override
    public List<User> getRecentlyCreatedUsers() throws TException {
        List users = Lists.create();
        Statement query = this.createQueryToGetRecentlyCreatedUsers();
        ResultSet results = this.tryToExecute(query);
        for (Row row : results) {
            User user = this.convertRowToUser(row);
            users.add(user);
        }
        return users;
    }

    private Statement createStatementToSaveUser(User user) {
        BatchStatement batchStatement = new BatchStatement(BatchStatement.Type.LOGGED);
        Statement insertIntoUsersTable = this.createInsertIntoUserTable(user);
        batchStatement.add(insertIntoUsersTable);
        Statement insertIntoUsersByEmailTable = this.createInsertIntoUsersByEmailTable(user);
        batchStatement.add(insertIntoUsersByEmailTable);
        if (!RequestAssertions.isNullOrEmpty(user.githubProfile)) {
            Statement insertIntoUsersByGithubTable = this.createInsertIntoUsersByGithubTable(user);
            batchStatement.add(insertIntoUsersByGithubTable);
        }
        Statement insertIntoRecentUsersTable = this.createInsertIntoRecentUsersTable(user);
        batchStatement.add(insertIntoRecentUsersTable);
        return batchStatement;
    }

    private Statement createInsertIntoUserTable(User user) {
        UUID userUuid = UUID.fromString(user.userId);
        Set emails = Sets.createFrom((Object)user.email, (Object[])new String[0]);
        return this.queryBuilder.insertInto("Users").value("user_id", (Object)userUuid).value("first_name", (Object)user.firstName).value("middle_name", (Object)user.middleName).value("last_name", (Object)user.lastName).value("emails", (Object)emails).value("roles", (Object)user.roles).value("github_profile", (Object)user.githubProfile).value("profile_image_id", (Object)user.profileImageLink).value("time_account_created", (Object)Instant.now().toEpochMilli());
    }

    private Statement createInsertIntoRecentUsersTable(User user) {
        UUID userUuid = UUID.fromString(user.userId);
        Set emails = Sets.createFrom((Object)user.email, (Object[])new String[0]);
        LengthOfTime timeUserIsRecent = new LengthOfTime().setUnit(TimeUnit.DAYS).setValue(3L);
        int recentDuration = (int)TimeFunctions.toSeconds((LengthOfTime)timeUserIsRecent);
        return this.queryBuilder.insertInto("Users_Recent").value("user_id", (Object)userUuid).value("first_name", (Object)user.firstName).value("middle_name", (Object)user.middleName).value("last_name", (Object)user.lastName).value("emails", (Object)emails).value("roles", (Object)user.roles).value("github_profile", (Object)user.githubProfile).value("profile_image_id", (Object)user.profileImageLink).value("time_account_created", (Object)Instant.now().toEpochMilli()).using(QueryBuilder.ttl((int)recentDuration));
    }

    private Statement createInsertIntoUsersByEmailTable(User user) {
        UUID userUuid = UUID.fromString(user.userId);
        return this.queryBuilder.insertInto("Users_By_Email").value("user_id", (Object)userUuid).value("email", (Object)user.email).value("first_name", (Object)user.firstName).value("middle_name", (Object)user.middleName).value("last_name", (Object)user.lastName).value("github_profile", (Object)user.githubProfile).value("profile_image_id", (Object)user.profileImageLink).value("time_account_created", (Object)Instant.now().toEpochMilli());
    }

    private Statement createInsertIntoUsersByGithubTable(User user) {
        UUID userUuid = UUID.fromString(user.userId);
        return this.queryBuilder.insertInto("Users_By_Github_Profile").value("github_profile", (Object)user.githubProfile).value("user_id", (Object)userUuid).value("first_name", (Object)user.firstName).value("middle_name", (Object)user.middleName).value("last_name", (Object)user.lastName).value("email", (Object)user.email).value("time_account_created", (Object)Instant.now().toEpochMilli());
    }

    private Statement createQueryToGetUser(String userId) {
        UUID userUuid = UUID.fromString(userId);
        return this.queryBuilder.select().all().from("Users").where(QueryBuilder.eq((String)"user_id", (Object)userUuid)).limit(2);
    }

    private Statement createQueryToGetUserByEmail(String email) {
        return this.queryBuilder.select().all().from("Users_By_Email").where(QueryBuilder.eq((String)"email", (Object)email));
    }

    private Statement createQueryToGetUsersByGithubProfile(String githubProfile) {
        return this.queryBuilder.select().all().from("Users_By_Github_Profile").where(QueryBuilder.eq((String)"github_profile", (Object)githubProfile));
    }

    private Statement createQueryToGetRecentlyCreatedUsers() {
        return this.queryBuilder.select().all().from("Users_Recent").limit(100);
    }

    private User convertRowToUser(Row row) {
        return this.userMapper.apply(row);
    }

    private Statement createQueryToDeleteUser(User user) {
        UUID userUuuid = UUID.fromString(user.userId);
        BatchStatement batch = new BatchStatement();
        Delete.Where deleteFromUsersTable = this.queryBuilder.delete().all().from("Users").where(QueryBuilder.eq((String)"user_id", (Object)userUuuid));
        batch.add((Statement)deleteFromUsersTable);
        Delete.Where deleteFromRecentUsersTable = this.queryBuilder.delete().all().from("Users_Recent").where(QueryBuilder.eq((String)"user_id", (Object)userUuuid));
        batch.add((Statement)deleteFromRecentUsersTable);
        Delete.Where deleteFromUserEmailsTable = this.queryBuilder.delete().all().from("Users_By_Email").where(QueryBuilder.eq((String)"email", (Object)user.email));
        batch.add((Statement)deleteFromUserEmailsTable);
        if (!RequestAssertions.isNullOrEmpty(user.githubProfile)) {
            Delete.Where deleteFromGithubTable = this.queryBuilder.delete().all().from("Users_By_Github_Profile").where(QueryBuilder.eq((String)"github_profile", (Object)user.githubProfile));
            batch.add((Statement)deleteFromGithubTable);
        }
        return batch;
    }

    private Statement createQueryToCheckExistenceFor(String userId) {
        UUID userUuid = UUID.fromString(userId);
        return this.queryBuilder.select().countAll().from("Users").where(QueryBuilder.eq((String)"user_id", (Object)userUuid));
    }

    private void checkUserId(String userId) throws InvalidArgumentException {
        Arguments.checkThat((Object)userId).throwing(InvalidArgumentException.class).usingMessage("missing userId").is(StringAssertions.nonEmptyString()).usingMessage("expecting UUID for userId").is(StringAssertions.validUUID());
    }

    private ResultSet tryToExecute(Statement statement) throws OperationFailedException {
        try {
            return this.cassandra.execute(statement);
        }
        catch (Exception ex) {
            LOG.error("Cassandra Operation failed", (Throwable)ex);
            throw new OperationFailedException("Data Operation Failed: " + ex.getMessage());
        }
    }
}

