/*
 * Decompiled with CFR 0.152.
 */
package tech.aroma.data.cassandra;

import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.querybuilder.QueryBuilder;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.inject.Inject;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sir.wellington.alchemy.collections.lists.Lists;
import sir.wellington.alchemy.collections.sets.Sets;
import tech.aroma.data.OrganizationRepository;
import tech.aroma.data.assertions.RequestAssertions;
import tech.aroma.thrift.Organization;
import tech.aroma.thrift.User;
import tech.aroma.thrift.exceptions.InvalidArgumentException;
import tech.aroma.thrift.exceptions.OperationFailedException;
import tech.aroma.thrift.exceptions.OrganizationDoesNotExistException;
import tech.sirwellington.alchemy.arguments.Arguments;
import tech.sirwellington.alchemy.arguments.assertions.Assertions;

final class CassandraOrganizationRepository
implements OrganizationRepository {
    private static final Logger LOG = LoggerFactory.getLogger(CassandraOrganizationRepository.class);
    private final Session cassandra;
    private final QueryBuilder queryBuilder;
    private final Function<Row, Organization> organizationMapper;
    private final Function<Row, User> userMapper;

    @Inject
    CassandraOrganizationRepository(Session cassandra, QueryBuilder queryBuilder, Function<Row, Organization> organizationMapper, Function<Row, User> userMapper) {
        Arguments.checkThat((Object)cassandra, (Object[])new Object[]{queryBuilder, organizationMapper, userMapper}).are(Assertions.notNull());
        this.cassandra = cassandra;
        this.queryBuilder = queryBuilder;
        this.organizationMapper = organizationMapper;
        this.userMapper = userMapper;
    }

    @Override
    public void saveOrganization(Organization organization) throws TException {
        Arguments.checkThat((Object)organization).throwing(InvalidArgumentException.class).is(RequestAssertions.validOrganization());
        Statement insertStatement = this.createStatementToInsert(organization);
        try {
            this.cassandra.execute(insertStatement);
        }
        catch (Exception ex) {
            LOG.error("Failed to save Organization in Cassandra: [{}]", (Object)organization, (Object)ex);
            throw new OperationFailedException("Failed to save Organization: " + ex.getMessage());
        }
    }

    @Override
    public Organization getOrganization(String organizationId) throws TException {
        ResultSet results;
        this.checkOrganizationId(organizationId);
        Statement query = this.createQueryToGetOrganization(organizationId);
        try {
            results = this.cassandra.execute(query);
        }
        catch (Exception ex) {
            LOG.error("Failed to query for Organization [{}]", (Object)organizationId, (Object)ex);
            throw new OperationFailedException("Could not get Organization: " + ex.getMessage());
        }
        Row row = results.one();
        this.checkRowIsPresent(row);
        Organization org = this.organizationMapper.apply(row);
        return org;
    }

    @Override
    public void deleteOrganization(String organizationId) throws TException {
        this.deleteAllMembers(organizationId);
        Statement deleteStatement = this.createStatementToDelete(organizationId);
        try {
            this.cassandra.execute(deleteStatement);
        }
        catch (Exception ex) {
            LOG.error("Failed to delete Organization in Cassandra: [{}]", (Object)organizationId, (Object)ex);
            throw new OperationFailedException("Failed to delete Organization: " + ex.getMessage());
        }
    }

    @Override
    public boolean containsOrganization(String organizationId) throws TException {
        ResultSet results;
        this.checkOrganizationId(organizationId);
        Statement query = this.createQueryToCheckIfOrgExists(organizationId);
        try {
            results = this.cassandra.execute(query);
        }
        catch (Exception ex) {
            LOG.error("Failed to query for existence of Organization [{}]", (Object)organizationId, (Object)ex);
            throw new OperationFailedException("Could not query for Organization: " + ex.getMessage());
        }
        Row row = results.one();
        this.checkRowIsPresent(row);
        long count = row.getLong(0);
        return count > 0L;
    }

    @Override
    public List<User> getOrganizationMembers(String organizationId) throws TException {
        ResultSet results;
        this.checkOrganizationId(organizationId);
        Statement query = this.createQueryToGetOrganizationMembers(organizationId);
        try {
            results = this.cassandra.execute(query);
        }
        catch (Exception ex) {
            LOG.error("Failed to query for Organization Members: [{}]", (Object)organizationId, (Object)ex);
            throw new OperationFailedException("Could not query for Organization Members: " + ex.getMessage());
        }
        List members = Lists.create();
        for (Row row : results) {
            User member = this.userMapper.apply(row);
            members.add(member);
        }
        LOG.debug("Found {} members in Org [{]]", (Object)members.size(), (Object)organizationId);
        return members;
    }

    @Override
    public List<Organization> searchByName(String searchTerm) throws TException {
        throw new OperationFailedException("Organization Search not supported yet.");
    }

    @Override
    public void saveMemberInOrganization(String organizationId, User user) throws TException {
        this.checkOrganizationId(organizationId);
        Arguments.checkThat((Object)user).throwing(InvalidArgumentException.class).is(RequestAssertions.validUser());
        Statement insertStatement = this.createStatementToSaveMember(organizationId, user);
        try {
            this.cassandra.execute(insertStatement);
        }
        catch (Exception ex) {
            LOG.error("Failed to save Member [{}] in Organization [{}] in Cassandra: [{}]", new Object[]{user, organizationId, ex});
            throw new OperationFailedException("Failed to save Member in Organization: " + ex.getMessage());
        }
    }

    @Override
    public boolean isMemberInOrganization(String organizationId, String userId) throws TException {
        ResultSet results;
        this.checkOrganizationId(organizationId);
        this.checkUserId(userId);
        Statement query = this.createQueryToSeeIfMemberOfOrg(organizationId, userId);
        try {
            results = this.cassandra.execute(query);
        }
        catch (Exception ex) {
            LOG.error("Failed to query for existence of Member [{]] in Organization [{}]", new Object[]{userId, organizationId, ex});
            throw new OperationFailedException("Could not query for membership in Organization: " + ex.getMessage());
        }
        Row row = results.one();
        this.checkRowIsPresent(row);
        long count = row.getLong(0);
        return count > 0L;
    }

    @Override
    public List<User> getOrganizationOwners(String organizationId) throws TException {
        Organization org = this.getOrganization(organizationId);
        Set owners = Sets.copyOf((Collection)org.owners);
        return owners.stream().map(id -> new User().setUserId(id)).collect(Collectors.toList());
    }

    @Override
    public void deleteMember(String organizationId, String userId) throws TException {
        this.checkOrganizationId(organizationId);
        this.checkUserId(userId);
        Statement deleteStatement = this.createStatmentToDeleteMember(organizationId, userId);
        try {
            this.cassandra.execute(deleteStatement);
        }
        catch (Exception ex) {
            LOG.error("Failed to delete Member [{}] in Organization [{}]", new Object[]{userId, organizationId, ex});
            throw new OperationFailedException("Failed to delete Member in Organization: " + ex.getMessage());
        }
    }

    @Override
    public void deleteAllMembers(String organizationId) throws TException {
        this.checkOrganizationId(organizationId);
        Statement deleteStatement = this.createStatementToDeleteAllMembers(organizationId);
        try {
            this.cassandra.execute(deleteStatement);
        }
        catch (Exception ex) {
            LOG.error("Failed to delete all Members in Organization [{}]", (Object)organizationId, (Object)ex);
            throw new OperationFailedException("Failed to delete all members in Organization: " + ex.getMessage());
        }
    }

    private Statement createStatementToInsert(Organization org) {
        UUID orgUuid = UUID.fromString(org.organizationId);
        Set owners = Lists.nullToEmpty((List)org.owners).stream().map(UUID::fromString).collect(Collectors.toSet());
        return this.queryBuilder.insertInto("Organizations").value("org_id", (Object)orgUuid).value("org_name", (Object)org.organizationName).value("owners", owners).value("icon_link", (Object)org.logoLink).value("industry", (Object)org.industry).value("contact_email", (Object)org.organizationEmail).value("github_profile", (Object)org.githubProfile).value("stock_name", (Object)org.stockMarketSymbol).value("tier", (Object)org.tier).value("description", (Object)org.organizationDescription).value("website", (Object)org.website);
    }

    private Statement createQueryToGetOrganization(String organizationId) {
        UUID orgUuid = UUID.fromString(organizationId);
        return this.queryBuilder.select().all().from("Organizations").where(QueryBuilder.eq((String)"org_id", (Object)orgUuid));
    }

    private void checkRowIsPresent(Row row) throws OperationFailedException, OrganizationDoesNotExistException {
        Arguments.checkThat((Object)row).throwing(OrganizationDoesNotExistException.class).usingMessage("Org Does not exist").is(Assertions.notNull());
    }

    private Statement createStatementToDelete(String organizationId) {
        UUID orgUuid = UUID.fromString(organizationId);
        return this.queryBuilder.delete().all().from("Organizations").where(QueryBuilder.eq((String)"org_id", (Object)orgUuid));
    }

    private Statement createQueryToCheckIfOrgExists(String organizationId) {
        UUID orgUuid = UUID.fromString(organizationId);
        return this.queryBuilder.select().countAll().from("Organizations").where(QueryBuilder.eq((String)"org_id", (Object)orgUuid));
    }

    private Statement createQueryToGetOrganizationMembers(String organizationId) {
        UUID orgUuid = UUID.fromString(organizationId);
        return this.queryBuilder.select().column("user_id").as("user_id").column("user_first_name").as("first_name").column("user_middle_name").as("middle_name").column("user_last_name").as("last_name").column("user_roles").as("roles").column("user_email").as("email").from("Organizations_Members").where(QueryBuilder.eq((String)"org_id", (Object)orgUuid));
    }

    private Statement createStatementToSaveMember(String organizationId, User user) {
        UUID orgUuid = UUID.fromString(organizationId);
        UUID userUuid = UUID.fromString(user.userId);
        return this.queryBuilder.insertInto("Organizations_Members").value("org_id", (Object)orgUuid).value("user_id", (Object)userUuid).value("user_first_name", (Object)user.firstName).value("user_last_name", (Object)user.lastName).value("user_roles", (Object)user.roles).value("user_email", (Object)user.email);
    }

    private Statement createQueryToSeeIfMemberOfOrg(String organizationId, String userId) {
        UUID orgUuid = UUID.fromString(organizationId);
        UUID userUuid = UUID.fromString(userId);
        return this.queryBuilder.select().countAll().from("Organizations_Members").where(QueryBuilder.eq((String)"org_id", (Object)orgUuid)).and(QueryBuilder.eq((String)"user_id", (Object)userUuid));
    }

    private Statement createStatmentToDeleteMember(String organizationId, String userId) {
        UUID orgUuid = UUID.fromString(organizationId);
        UUID userUuid = UUID.fromString(userId);
        return this.queryBuilder.delete().all().from("Organizations_Members").where(QueryBuilder.eq((String)"org_id", (Object)orgUuid)).and(QueryBuilder.eq((String)"user_id", (Object)userUuid));
    }

    private Statement createStatementToDeleteAllMembers(String organizationId) {
        UUID orgUuid = UUID.fromString(organizationId);
        return this.queryBuilder.delete().all().from("Organizations_Members").where(QueryBuilder.eq((String)"org_id", (Object)orgUuid));
    }

    private void checkOrganizationId(String organizationId) throws InvalidArgumentException {
        Arguments.checkThat((Object)organizationId).throwing(InvalidArgumentException.class).is(RequestAssertions.validOrgId());
    }

    private void checkUserId(String userId) throws InvalidArgumentException {
        Arguments.checkThat((Object)userId).throwing(InvalidArgumentException.class).is(RequestAssertions.validUserId());
    }
}

