/*
 * Decompiled with CFR 0.152.
 */
package tech.aroma.data.cassandra;

import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.querybuilder.Ordering;
import com.datastax.driver.core.querybuilder.QueryBuilder;
import java.util.List;
import java.util.UUID;
import java.util.function.Function;
import javax.inject.Inject;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sir.wellington.alchemy.collections.lists.Lists;
import tech.aroma.data.InboxRepository;
import tech.aroma.data.assertions.RequestAssertions;
import tech.aroma.thrift.LengthOfTime;
import tech.aroma.thrift.Message;
import tech.aroma.thrift.User;
import tech.aroma.thrift.exceptions.InvalidArgumentException;
import tech.aroma.thrift.exceptions.OperationFailedException;
import tech.aroma.thrift.functions.TimeFunctions;
import tech.sirwellington.alchemy.annotations.arguments.Required;
import tech.sirwellington.alchemy.arguments.Arguments;
import tech.sirwellington.alchemy.arguments.assertions.Assertions;

final class CassandraInboxRepository
implements InboxRepository {
    private static final Logger LOG = LoggerFactory.getLogger(CassandraInboxRepository.class);
    private final Session cassandra;
    private final QueryBuilder queryBuilder;
    private final Function<Row, Message> messageMapper;

    @Inject
    CassandraInboxRepository(Session cassandra, QueryBuilder queryBuilder, Function<Row, Message> messageMapper) {
        Arguments.checkThat((Object)cassandra, (Object[])new Object[]{queryBuilder, messageMapper}).are(Assertions.notNull());
        this.cassandra = cassandra;
        this.queryBuilder = queryBuilder;
        this.messageMapper = messageMapper;
    }

    @Override
    public void saveMessageForUser(@Required User user, @Required Message message, @Required LengthOfTime lifetime) throws TException {
        Arguments.checkThat((Object)message).throwing(ex -> new InvalidArgumentException(ex.getMessage())).is(RequestAssertions.validMessage());
        Arguments.checkThat((Object)user).throwing(ex -> new InvalidArgumentException(ex.getMessage())).is(RequestAssertions.validUser());
        Statement insertStatement = this.createStatementToSaveMessage(message, user, lifetime);
        try {
            this.cassandra.execute(insertStatement);
        }
        catch (Exception ex2) {
            LOG.error("Failed to save Message in Cassandra Inbox. User [{}] Message [{}]", new Object[]{user.userId, message, ex2});
            throw new OperationFailedException("Could not save message in Inbox: " + ex2.getMessage());
        }
    }

    @Override
    public List<Message> getMessagesForUser(String userId) throws TException {
        ResultSet results;
        Arguments.checkThat((Object)userId).throwing(InvalidArgumentException.class).is(RequestAssertions.validUserId());
        Statement query = this.createQueryToGetMessagesFor(userId);
        try {
            results = this.cassandra.execute(query);
        }
        catch (Exception ex) {
            LOG.error("Failed to query for Messages in Inbox for User [{}]", (Object)userId, (Object)ex);
            throw new OperationFailedException("Could not fetch inbox: " + ex.getMessage());
        }
        Arguments.checkThat((Object)results).throwing(OperationFailedException.class).usingMessage("Cassandra returned null results").is(Assertions.notNull());
        List messages = Lists.create();
        for (Row row : results) {
            Message message = this.messageMapper.apply(row);
            messages.add(message);
        }
        return messages;
    }

    @Override
    public boolean containsMessageInInbox(String userId, Message message) throws TException {
        ResultSet results;
        this.checkUserId(userId);
        Arguments.checkThat((Object)message).throwing(InvalidArgumentException.class).is(RequestAssertions.validMessage());
        Statement query = this.createQueryToCheckIfInInboxOf(userId, message);
        try {
            results = this.cassandra.execute(query);
        }
        catch (Exception ex) {
            LOG.error("Failed to query Cassandra for presence of message [{}] for User [{}]", new Object[]{message.messageId, userId, ex});
            throw new OperationFailedException("Could not check if message exists: " + ex.getMessage());
        }
        Row row = results.one();
        Arguments.checkThat((Object)row).throwing(OperationFailedException.class).usingMessage("Query for message failed").is(Assertions.notNull());
        long count = row.getLong(0);
        return count > 0L;
    }

    @Override
    public void deleteMessageForUser(String userId, String messageId) throws TException {
        this.checkUserId(userId);
        this.checkMessageId(messageId);
        Statement deleteStatement = this.createDeleteStatementFor(userId, messageId);
        try {
            this.cassandra.execute(deleteStatement);
        }
        catch (Exception ex) {
            LOG.error("Failed to delete message [{}] for User [{}] from Inbox", new Object[]{messageId, userId, ex});
            throw new OperationFailedException("Could not delete message: " + ex.getMessage());
        }
    }

    @Override
    public void deleteAllMessagesForUser(String userId) throws TException {
        this.checkUserId(userId);
        Statement deleteStatement = this.createDeleteAllStatementFor(userId);
        try {
            this.cassandra.execute(deleteStatement);
        }
        catch (Exception ex) {
            LOG.error("Failed to delete all messages for User [{}] from Inbox", (Object)userId, (Object)ex);
            throw new OperationFailedException("Could not delete message: " + ex.getMessage());
        }
    }

    @Override
    public long countInboxForUser(String userId) throws TException {
        ResultSet results;
        this.checkUserId(userId);
        Statement query = this.createQueryToCountMessagesFor(userId);
        try {
            results = this.cassandra.execute(query);
        }
        catch (Exception ex) {
            LOG.error("Failed to count total messages for User [{}]", (Object)userId, (Object)ex);
            throw new OperationFailedException("Could not count messags for user: " + ex.getMessage());
        }
        Row row = results.one();
        Arguments.checkThat((Object)row).throwing(OperationFailedException.class).usingMessage("Query for message failed").is(Assertions.notNull());
        long count = row.getLong(0);
        return count;
    }

    private Statement createStatementToSaveMessage(Message message, User user, LengthOfTime lifetime) {
        UUID msgUuid = UUID.fromString(message.messageId);
        UUID userUuid = UUID.fromString(user.userId);
        UUID appUuid = UUID.fromString(message.applicationId);
        long timeToLive = TimeFunctions.toSeconds((LengthOfTime)lifetime);
        return this.queryBuilder.insertInto("Inbox").value("user_id", (Object)userUuid).value("message_id", (Object)msgUuid).value("body", (Object)message.body).value("app_id", (Object)appUuid).value("urgency", (Object)message.urgency).value("title", (Object)message.title).value("time_created", (Object)message.timeOfCreation).value("time_received", (Object)message.timeMessageReceived).value("hostname", (Object)message.hostname).value("mac_address", (Object)message.macAddress).value("app_name", (Object)message.applicationName).using(QueryBuilder.ttl((int)((int)timeToLive)));
    }

    private Statement createQueryToGetMessagesFor(String userId) {
        UUID userUuid = UUID.fromString(userId);
        return this.queryBuilder.select().all().from("Inbox").where(QueryBuilder.eq((String)"user_id", (Object)userUuid)).orderBy(new Ordering[]{QueryBuilder.desc((String)"message_id")}).limit(5000);
    }

    private Statement createQueryToCheckIfInInboxOf(String userId, Message message) {
        UUID userUuid = UUID.fromString(userId);
        UUID msgUuid = UUID.fromString(message.messageId);
        return this.queryBuilder.select().countAll().from("Inbox").where(QueryBuilder.eq((String)"user_id", (Object)userUuid)).and(QueryBuilder.eq((String)"message_id", (Object)msgUuid));
    }

    private Statement createDeleteStatementFor(String userId, String messageId) {
        UUID userUuid = UUID.fromString(userId);
        UUID msgUuid = UUID.fromString(messageId);
        return this.queryBuilder.delete().all().from("Inbox").where(QueryBuilder.eq((String)"user_id", (Object)userUuid)).and(QueryBuilder.eq((String)"message_id", (Object)msgUuid));
    }

    private Statement createDeleteAllStatementFor(String userId) {
        UUID userUuid = UUID.fromString(userId);
        return this.queryBuilder.delete().all().from("Inbox").where(QueryBuilder.eq((String)"user_id", (Object)userUuid));
    }

    private Statement createQueryToCountMessagesFor(String userId) {
        UUID userUuid = UUID.fromString(userId);
        return this.queryBuilder.select().countAll().from("Inbox").where(QueryBuilder.eq((String)"user_id", (Object)userUuid));
    }

    private void checkMessageId(String messageId) throws InvalidArgumentException {
        Arguments.checkThat((Object)messageId).throwing(InvalidArgumentException.class).is(RequestAssertions.validMessageId());
    }

    private void checkUserId(String userId) throws InvalidArgumentException {
        Arguments.checkThat((Object)userId).throwing(ex -> new InvalidArgumentException(ex.getMessage())).is(RequestAssertions.validUserId());
    }
}

