/**
 * <h2>OpenAPI Gateway</h2>
 * <p>
 * Define your APIs using <a href="https://swagger.io/specification/">OpenAPI v3</a>, and leverage the power of generated clients, automatic input validation, and type safe client and server code!
 * <p>
 * This package vends a projen project type which allows you to define an API using <a href="https://swagger.io/specification/">OpenAPI v3</a>, and a construct which manages deploying this API in API Gateway, given a lambda integration for every operation.
 * <p>
 * The project will generate models and clients from your OpenAPI spec in your desired languages, and can be utilised both client side or server side in lambda handlers. The project type also generates a wrapper construct which adds type safety to ensure a lambda integration is provided for every API integration.
 * <p>
 * <h3>Project</h3>
 * <p>
 * It's recommended that this project is used as part of an <code>nx_monorepo</code> project. You can still use this as a standalone project if you like (eg <code>npx projen new --from &#64;aws-prototyping-sdk/open-api-gateway open-api-gateway-ts</code>), however you will need to manage build order (ie building the generated client first, followed by the project).
 * <p>
 * For usage in a monorepo:
 * <p>
 * Create the project in your .projenrc:
 * <p>
 * <blockquote><pre>
 * import {ClientLanguage, OpenApiGatewayTsProject} from "&#64;aws-prototyping-sdk/open-api-gateway";
 * 
 * new OpenApiGatewayTsProject({
 *   parent: myNxMonorepo,
 *   defaultReleaseBranch: "mainline",
 *   name: "my-api",
 *   outdir: "packages/api",
 *   clientLanguages: [ClientLanguage.TYPESCRIPT, ClientLanguage.PYTHON, ClientLanguage.JAVA],
 * });
 * </pre></blockquote>
 * <p>
 * In the output directory (<code>outdir</code>), you'll find a few files to get you started.
 * <p>
 * <blockquote><pre>
 * |_ src/
 *     |_ spec/
 *         |_ spec.yaml - The OpenAPI specification - edit this to define your API
 *         |_ .parsed-spec.json - A json spec generated from your spec.yaml.
 *     |_ api/
 *         |_ api.ts - A CDK construct which defines the API Gateway resources to deploy your API.
 *         |           This wraps the OpenApiGatewayLambdaApi construct and provides typed interfaces for integrations specific
 *         |           to your API. You shouldn't need to modify this, instead just extend it as in sample-api.ts.
 *         |_ sample-api.ts - Example usage of the construct defined in api.ts.
 *         |_ sample-api.say-hello.ts - An example lambda handler for the operation defined in spec.yaml, making use of the
 *                                      generated lambda handler wrappers for marshalling and type safety.
 * |_ generated/
 *     |_ typescript/ - A generated typescript API client, including generated lambda handler wrappers
 *     |_ python/ - A generated python API client.
 *     |_ java/ - A generated java API client.
 * </pre></blockquote>
 * <p>
 * If you would prefer to not generate the sample code, you can pass <code>sampleCode: false</code> to <code>OpenApiGatewayTsProject</code>.
 * <p>
 * To make changes to your api, simply update <code>spec.yaml</code> and run <code>npx projen</code> to synthesize all the typesafe client/server code!
 * <p>
 * <h3>OpenAPI Specification</h3>
 * <p>
 * Your <code>spec.yaml</code> file defines your api using <a href="https://swagger.io/specification/">OpenAPI Version 3.0.3</a>. An example spec might look like:
 * <p>
 * <blockquote><pre>
 * openapi: 3.0.3
 * info:
 *   version: 1.0.0
 *   title: Example API
 * paths:
 *   /hello:
 *     get:
 *       operationId: sayHello
 *       parameters:
 *         - in: query
 *           name: name
 *           schema:
 *             type: string
 *           required: true
 *       responses:
 *         '200':
 *           description: Successful response
 *           content:
 *             'application/json':
 *               schema:
 *                 $ref: '#/components/schemas/HelloResponse'
 * components:
 *   schemas:
 *     HelloResponse:
 *       type: object
 *       properties:
 *         message:
 *           type: string
 *       required:
 *         - message
 * </pre></blockquote>
 * <p>
 * You can divide your specification into multiple files using <code>$ref</code>.
 * <p>
 * For example, you might choose to structure your spec as follows:
 * <p>
 * <blockquote><pre>
 * |_ spec/
 *     |_ spec.yaml
 *     |_ paths/
 *         |_ index.yaml
 *         |_ sayHello.yaml
 *     |_ schemas/
 *         |_ index.yaml
 *         |_ helloResponse.yaml
 * </pre></blockquote>
 * <p>
 * Where <code>spec.yaml</code> looks as follows:
 * <p>
 * <blockquote><pre>
 * openapi: 3.0.3
 * info:
 *   version: 1.0.0
 *   title: Example API
 * paths:
 *   $ref: './paths/index.yaml'
 * components:
 *   schemas:
 *     $ref: './schemas/index.yaml'
 * </pre></blockquote>
 * <p>
 * <code>paths/index.yaml</code>:
 * <p>
 * <blockquote><pre>
 * /hello:
 *   get:
 *     $ref: './sayHello.yaml''\
 * </pre></blockquote>
 * <p>
 * <code>paths/sayHello.yaml</code>:
 * <p>
 * <blockquote><pre>
 * operationId: sayHello
 * parameters:
 *  - in: query
 *    name: name
 *    schema:
 *      type: string
 *    required: true
 * responses:
 *   '200':
 *     description: Successful response
 *     content:
 *       'application/json':
 *         schema:
 *           $ref: '../schemas/helloResponse.yaml'
 * </pre></blockquote>
 * <p>
 * <code>schemas/index.yaml</code>:
 * <p>
 * <blockquote><pre>
 * HelloResponse:
 *   $ref: './helloResponse.yaml'
 * </pre></blockquote>
 * <p>
 * <code>schemas/helloResponse.yaml</code>:
 * <p>
 * <blockquote><pre>
 * type: object
 * properties:
 *   message:
 *     type: string
 * required:
 *   - message
 * </pre></blockquote>
 * <p>
 * <h3>Construct</h3>
 * <p>
 * A sample construct is generated which provides a type-safe interface for creating an API Gateway API based on your OpenAPI specification. You'll get a type error if you forget to define an integration for an operation defined in your api.
 * <p>
 * <blockquote><pre>
 * import * as path from 'path';
 * import { Authorizers } from '&#64;aws-prototyping-sdk/open-api-gateway';
 * import { NodejsFunction } from 'aws-cdk-lib/aws-lambda-nodejs';
 * import { Construct } from 'constructs';
 * import { Api } from './api';
 * 
 * /**
 *  * An example of how to wire lambda handler functions to the API
 *  *&#47;
 * export class SampleApi extends Api {
 *   constructor(scope: Construct, id: string) {
 *     super(scope, id, {
 *       defaultAuthorizer: Authorizers.iam(),
 *       integrations: {
 *         // Every operation defined in your API must have an integration defined!
 *         sayHello: {
 *           function: new NodejsFunction(scope, 'say-hello'),
 *         },
 *       },
 *     });
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * <h4>Authorizers</h4>
 * <p>
 * The <code>Api</code> construct allows you to define one or more authorizers for securing your API. An integration will use the <code>defaultAuthorizer</code> unless an <code>authorizer</code> is specified at the integration level. The following authorizers are supported:
 * <p>
 * <ul>
 * <li><code>Authorizers.none</code> - No auth</li>
 * <li><code>Authorizers.iam</code> - AWS IAM (Signature Version 4)</li>
 * <li><code>Authorizers.cognito</code> - Cognito user pool</li>
 * <li><code>Authorizers.custom</code> - A custom authorizer</li>
 * </ul>
 * <p>
 * <h5>Cognito Authorizer</h5>
 * <p>
 * To use the Cognito authorizer, one or more user pools must be provided. You can optionally specify the scopes to check if using an access token. You can use the <code>withScopes</code> method to use the same authorizer but verify different scopes for individual integrations, for example:
 * <p>
 * <blockquote><pre>
 * export class SampleApi extends Api {
 *   constructor(scope: Construct, id: string) {
 *     const cognitoAuthorizer = Authorizers.cognito({
 *       authorizerId: 'myCognitoAuthorizer',
 *       userPools: [new UserPool(scope, 'UserPool')],
 *     });
 * 
 *     super(scope, id, {
 *       defaultAuthorizer: cognitoAuthorizer,
 *       integrations: {
 *         // Everyone in the user pool can call this operation:
 *         sayHello: {
 *           function: new NodejsFunction(scope, 'say-hello'),
 *         },
 *         // Only users with the given scopes can call this operation
 *         myRestrictedOperation: {
 *           function: new NodejsFunction(scope, 'my-restricted-operation'),
 *           authorizer: cognitoAuthorizer.withScopes('my-resource-server/my-scope'),
 *         },
 *       },
 *     });
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * For more information about scopes or identity and access tokens, please see the <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html">API Gateway documentation</a>.
 * <p>
 * <h5>Custom Authorizer</h5>
 * <p>
 * Custom authorizers use lambda functions to handle authorizing requests. These can either be simple token-based authorizers, or more complex request-based authorizers. See the <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html">API Gateway documentation</a> for more details.
 * <p>
 * An example token-based authorizer (default):
 * <p>
 * <blockquote><pre>
 * Authorizers.custom({
 *   authorizerId: 'myTokenAuthorizer',
 *   function: new NodejsFunction(scope, 'authorizer'),
 * });
 * </pre></blockquote>
 * <p>
 * An example request-based handler. By default the identitySource will be <code>method.request.header.Authorization</code>, however you can customise this as per <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-identitysource">the API Gateway documentation</a>.
 * <p>
 * <blockquote><pre>
 * Authorizers.custom({
 *   authorizerId: 'myRequestAuthorizer',
 *   type: CustomAuthorizerType.REQUEST,
 *   identitySource: 'method.request.header.MyCustomHeader, method.request.querystring.myQueryString',
 *   function: new NodejsFunction(scope, 'authorizer'),
 * });
 * </pre></blockquote>
 * <p>
 * <h3>Generated Client</h3>
 * <p>
 * <h4>Typescript</h4>
 * <p>
 * The <a href="https://openapi-generator.tech/docs/generators/typescript-fetch/">typescript-fetch</a> OpenAPI generator is used to generate OpenAPI clients for typescript. This requires an implementation of <code>fetch</code> to be passed to the client. In the browser one can pass the built in fetch, or in NodeJS you can use an implementation such as <a href="https://www.npmjs.com/package/node-fetch">node-fetch</a>.
 * <p>
 * Example usage of the client in a website:
 * <p>
 * <blockquote><pre>
 * import { Configuration, DefaultApi } from "my-api-typescript-client";
 * 
 * const client = new DefaultApi(new Configuration({
 *   basePath: "https://xxxxxxxxxx.execute-api.ap-southeast-2.amazonaws.com",
 *   fetchApi: window.fetch.bind(window),
 * }));
 * 
 * await client.sayHello({ name: "Jack" });
 * </pre></blockquote>
 * <p>
 * <h4>Python</h4>
 * <p>
 * The <a href="https://openapi-generator.tech/docs/generators/python-experimental">python-experimental</a> OpenAPI generator is used to generate OpenAPI clients for python.
 * <p>
 * Example usage:
 * <p>
 * <blockquote><pre>
 * from my_api_python import ApiClient, Configuration
 * from my_api_python.api.default_api import DefaultApi
 * 
 * configuration = Configuration(
 *     host = "https://xxxxxxxxxx.execute-api.ap-southeast-2.amazonaws.com"
 * )
 * 
 * with ApiClient(configuration) as api_client:
 *     client = DefaultApi(api_client)
 * 
 *     client.say_hello(
 *         query_params={
 *             'name': "name_example",
 *         },
 *     )
 * </pre></blockquote>
 * <p>
 * You'll find details about how to use the python client in the README.md alongside your generated client.
 * <p>
 * <h3>Lambda Handler Wrappers</h3>
 * <p>
 * Lambda handler wrappers are also importable from the generated client. These provide input/output type safety, as well as allowing you to define a custom type for API error responses.
 * <p>
 * <blockquote><pre>
 * import { sayHelloHandler, ApiError } from "my-api-typescript-client";
 * 
 * export const handler = sayHelloHandler&lt;ApiError&gt;(async (input) =&gt; {
 *   return {
 *     statusCode: 200,
 *     body: {
 *       message: `Hello ${input.requestParameters.name}!`,
 *     },
 *   };
 * });
 * </pre></blockquote>
 * <p>
 * <h3>Other Details</h3>
 * <p>
 * <h4>Workspaces and <code>OpenApiGatewayTsProject</code></h4>
 * <p>
 * <code>OpenApiGatewayTsProject</code> can be used as part of a monorepo using YARN/NPM/PNPM workspaces. When used in a monorepo, a dependency is established between <code>OpenApiGatewayTsProject</code> and the generated typescript client, which is expected to be managed by the parent monorepo (ie both <code>OpenApiGatewayTsProject</code> and the generated typescript client are parented by the monorepo).
 * <p>
 * During initial project synthesis, the dependency between <code>OpenApiGatewayTsProject</code> and the generated client is established via workspace configuration local to <code>OpenApiGatewayTsProject</code>, since the parent monorepo will not have updated to include the new packages in time for the initial "install".
 * <p>
 * When the package manager is PNPM, this initial workspace is configured by creating a local <code>pnpm-workspace.yaml</code> file, and thus if you specify your own for an instance of <code>OpenApiGatewayTsProject</code>, synthesis will fail. It is most likely that you will not need to define this file locally in <code>OpenApiGatewayTsProject</code> since the monorepo copy should be used to manage all packages within the repo, however if managing this file at the <code>OpenApiGatewayTsProject</code> level is required, please use the <code>pnpmWorkspace</code> property of <code>OpenApiGatewayTsProject</code>.
 * <p>
 * <h4>Customising Generated Client Projects</h4>
 * <p>
 * By default, the generated clients are configured automatically, including project names. You can customise the generated client code using the <code>&lt;language&gt;ProjectOptions</code> properties when constructing your projen project.
 * <p>
 * <h5>Python Shared Virtual Environment</h5>
 * <p>
 * For adding dependencies between python projects within a monorepo you can use a single shared virtual environment, and install your python projects into that environment with <code>pip install --editable .</code> in the dependee. The generated python client will automatically do this if it detects it is within a monorepo.
 * <p>
 * The following example shows how to configure the generated client to use a shared virtual environment:
 * <p>
 * <blockquote><pre>
 * const api = new OpenApiGatewayTsProject({
 *   parent: monorepo,
 *   name: 'api',
 *   outdir: 'packages/api',
 *   defaultReleaseBranch: 'main',
 *   clientLanguages: [ClientLanguage.PYTHON],
 *   pythonClientOptions: {
 *     moduleName: 'my_api_python',
 *     name: 'my_api_python',
 *     authorName: 'jack',
 *     authorEmail: 'me&#64;example.com',
 *     version: '1.0.0',
 *     venvOptions: {
 *       // Directory relative to the generated python client (in this case packages/api/generated/python)
 *       envdir: '../../../../.env',
 *     },
 *   },
 * });
 * 
 * new PythonProject({
 *   parent: monorepo,
 *   outdir: 'packages/my-python-lib',
 *   moduleName: 'my_python_lib',
 *   name: 'my_python_lib',
 *   authorName: 'jack',
 *   authorEmail: 'me&#64;example.com',
 *   version: '1.0.0',
 *   venvOptions: {
 *     // Directory relative to the python lib (in this case packages/my-python-lib)
 *     envdir: '../../.env',
 *   },
 *   // Generated client can be safely cast to a PythonProject
 *   deps: [(api.generatedClients[ClientLanguage.PYTHON] as PythonProject).moduleName],
 * });
 * </pre></blockquote>
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Experimental)
package software.aws.awsprototypingsdk.openapigateway;
