package ru.curs.adocwrapper

import org.asciidoctor.Asciidoctor
import org.asciidoctor.Options
import org.w3c.dom.Document
import org.w3c.dom.NodeList
import org.xml.sax.InputSource
import ru.curs.adocwrapper.block.StructuralNode
import ru.curs.adocwrapper.block.image.Image
import ru.curs.adocwrapper.block.list.OList
import ru.curs.adocwrapper.block.list.UList
import ru.curs.adocwrapper.block.paragraph.Paragraph
import ru.curs.adocwrapper.block.section.Section
import ru.curs.adocwrapper.block.signature.Signature
import ru.curs.adocwrapper.block.table.Table
import java.io.StringReader
import javax.xml.parsers.DocumentBuilderFactory
import javax.xml.xpath.XPathConstants
import javax.xml.xpath.XPathFactory

object AsciidocValidatorFactory {
    private val factory: Asciidoctor = Asciidoctor.Factory.create()

    init {
        factory.requireLibrary(this::class.java.getResource("/asciiml.rb") !!.toString())
    }

    fun getXML(string: String): String {
        return factory.convert(string, Options.builder().backend("asciiml").sourcemap(true).build())
    }
}

object DBFactory {
    private val factory: DocumentBuilderFactory = DocumentBuilderFactory.newInstance()
    fun parseDocument(string: String): Document {
        return factory.newDocumentBuilder().parse(InputSource(StringReader(string)))
    }
}

object XPFactory {
    private val factory: XPathFactory = XPathFactory.newInstance()
    fun eval(xpath: String, document: Document): NodeList {
        return factory.newXPath().evaluate(xpath, document, XPathConstants.NODESET) as NodeList
    }
}

class AdocDocument(init: AdocDocument.() -> Unit) : StructuralNode() {
    var titleImageURL: String? = null

    init {
        sectionLevel = 0
        apply(init)
    }

    fun titleImage(url: String) {
        this.titleImageURL = url
    }

    override fun toString(): String {
        var returnString = getIdRoleSyntax()
        returnString += if (title == null) "" else "= $title"
        attrs.forEach { attr ->
            returnString += "\n:${attr.key}: ${attr.value}"
        }
        if (titleImageURL != null) {
            returnString += "\n\nimage::${titleImageURL}[]"
        }
        blocks.forEach { block ->
            returnString += block.toString()
        }
        return returnString
    }

    override fun toHabrMd(): String {
        var returnString = ""
        if (titleImageURL != null) {
            returnString += "![]($titleImageURL)"
        }
        blocks.forEach { block ->
            returnString += block.toHabrMd()
        }
        return returnString
    }

    override fun toText(): String {
        var returnString = ""
        returnString += if (title == null) "" else "$title"
        if (titleImageURL != null) {
            returnString += "\n\nTitle image URL: $titleImageURL)"
        }
        blocks.forEach { block ->
            returnString += block.toText()
        }
        return returnString
    }

    public override fun p(init: Paragraph.() -> Unit): Paragraph {
        return super.p(init)
    }

    public override fun ol(init: OList.() -> Unit): OList {
        return super.ol(init)
    }

    public override fun ul(init: UList.() -> Unit): UList {
        return super.ul(init)
    }

    public override fun section(init: Section.() -> Unit): Section {
        return super.section(init)
    }

    public override fun table(init: Table.() -> Unit): Table {
        return super.table(init)
    }

    public override fun image(init: Image.() -> Unit): Image {
        return super.image(init)
    }

    public override fun signature(init: Signature.() -> Unit): Signature {
        return super.signature(init)
    }

}

fun markupDocument(init: AdocDocument.() -> Unit): AdocDocument {
    return AdocDocument(init)
}
