/*
 * Copyright (c) 2021 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.metamodel.data.constraints

import pl.metaprogramming.metamodel.data.DataSchema

class Constraints {
    private static final String X_CONSTRAINTS = 'x-constraints'
    private static final String DICTIONARY = 'dictionary'
    private static final String INVALID_PATTERN_CODE = 'invalid-pattern-code'
    private DataSchema dataSchema

    Constraints(DataSchema dataSchema) {
        this.dataSchema = dataSchema
    }

    List<DataConstraint> getConstraints() {
        resetPriority()
        supportedConstraints.collect {
            fillCommons(makeConstraint(it), it)
        } as List<DataConstraint>
    }

    Constraints add(DictionaryConstraint constraint) {
        def spec = [dictionary: constraint.dictionaryCode]
        if (constraint.invalidCode) spec['invalid-code'] = constraint.invalidCode
        if (constraint.priority != null) spec.priority = constraint.priority
        getAdditives(X_CONSTRAINTS).add(spec)
        this
    }

    Constraints invalidPatternCode(String code) {
        getAdditives(X_CONSTRAINTS).add([(INVALID_PATTERN_CODE): code])
        this
    }

    String getInvalidPatternCode() {
        getAdditives(X_CONSTRAINTS)?.findResult {
            it instanceof Map && it.containsKey(INVALID_PATTERN_CODE) ? it[INVALID_PATTERN_CODE] : null
        } as String
    }

    private List getAdditives(String element) {
        if (!dataSchema.additives[element]) {
            dataSchema.additives[element] = []
        }
        dataSchema.additives[element] as List
    }

    private Collection<Map> getSupportedConstraints() {
        getAdditives(X_CONSTRAINTS)
                .findAll {
                    it instanceof Map && it.containsKey(DICTIONARY)
                } as List<Map>
    }

    private DataConstraint makeConstraint(Map spec) {
        new DictionaryConstraint(dictionaryCode: spec[DICTIONARY])
    }

    private DataConstraint fillCommons(DataConstraint constraint, Map spec) {
        constraint.invalidCode = spec['invalid-code']
        constraint.priority = spec.priority != null ? spec.priority as Integer : nextPriority()
        constraint
    }

    private Integer currentPriority

    private void resetPriority() {
        currentPriority = 100
    }

    private Integer nextPriority() {
        ++currentPriority
    }
}
