/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.model.java.index

import pl.metaprogramming.codemodel.model.java.ClassBuilder
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.builder.java.MethodCmBuilder
import pl.metaprogramming.codemodel.model.java.MethodCm

class ClassIndex {

    List<ClassIndex> dependsOn = []

    private final String name
    private final Map<ClassEntry.Key, ClassEntry> index = new HashMap<>()
    private final Map<MapperEntry.Key, MapperEntry> mappersIndex = [:]
    private final Set<String> usedNames = []
    private List orderedClassTypes

    ClassIndex(String name, List buildOrder = []) {
        this.name = name
        orderedClassTypes = recalculateOrderedClassTypes(buildOrder)
    }

    Collection<ClassEntry> getCodesToGenerate() {
        index.values().findAll { it.toGenerate }
    }

    void makeCodeModels() {
        orderedClassTypes = recalculateOrderedClassTypes(orderedClassTypes)
        makeCodeDeclarations()
        while (areCodeModelsToBuild()) {
            makeClasses()
            makeMappers()
        }
    }

    void makeCodeDeclarations() {
        orderedClassTypes.each { it ->
            findByClassType(it).each { it.builder.makeDeclaration() }
        }
    }

    boolean areCodeModelsToBuild() {
        index.values().any { it.toMake } ||
                mappersIndex.values().any { it.toMake } ||
                dependsOn.any { it.areCodeModelsToBuild() }
    }

    void makeClasses() {
        orderedClassTypes.each {
            findByClassType(it)
                    .findAll { it.toMake }
                    .each { it.makeImplementation() }
        }
        dependsOn.each { it.makeClasses() }
    }

    void makeMappers() {
        mappersIndex.values()
                .findAll { it.toMake }
                .each {
                    it.makeImplementation()
                }
        dependsOn.each {
            it.makeMappers()
        }
    }

    void put(ClassBuilder builder) {
        put(builder.classCd, builder.model, builder.config.classType, builder)
    }

    void put(ClassCd classCd, def model, def classType, ClassBuilder builder = null) {
        checkClassName(classCd.canonicalName, true)
        def entry = new ClassEntry(clazz: classCd, classType: classType, model: model, builder: builder)
        index.put(entry.key, entry)
    }

    ClassCd useClass(Object classType) {
        getClass(classType, null, true)
    }

    ClassCd getClass(Object classType, Object model, boolean markAsUsed) {
        def result = getClassByKey(ClassEntry.Key.of(classType, model), markAsUsed)
        result ?: model != null ? getClassByKey(ClassEntry.Key.of(classType), markAsUsed) : null
    }


    private ClassCd getClassByKey(ClassEntry.Key key, boolean markAsUsed) {
        def entry = index.get(key)
        if (entry != null) {
            if (markAsUsed) {
                entry.markAsUsed()
            }
            entry.clazz
        } else {
            dependsOn.findResult { it.getClassByKey(key, markAsUsed) }
        }
    }

    void putMappers(MethodCm... methods) {
        methods.each {
            addMapper(new MapperEntry(externalMethod: it))
        }
    }

    void putMapper(MethodCmBuilder mapperBuilder) {
        addMapper(new MapperEntry(
                builder: mapperBuilder,
                ownerClass: index.values().find { it.clazz == mapperBuilder.methodCm.ownerClass }))
    }

    private void addMapper(MapperEntry mapper) {
        if (mapper.methodCm.resultType == null) {
            throw new IllegalArgumentException("Mapper should return value")
        }
        if (mappersIndex.containsKey(mapper.key)) {
            printIndex()
            throw new IllegalStateException("Mapper already defined $mapper.key")

        }
        mappersIndex.put(mapper.key, mapper)
    }

    MapperEntry getMapper(MapperEntry.Key key) {
        def result = mappersIndex.get(key)
        if (result) {
            result.markAsUsed()
            result
        } else {
            dependsOn.findResult { it.getMapper(key) }
        }
    }

    private Collection<ClassEntry> findByClassType(Object classType) {
        index.values().findAll {
            it.classType == classType && it.builder != null
        }
    }

    void checkClassName(String canonicalName, boolean markAsUsed) {
        if (usedNames.contains(canonicalName)) {
            throw new IllegalStateException("class $canonicalName already registred")
        }
        dependsOn.each {it.checkClassName(canonicalName, false) }
        if (markAsUsed) {
            usedNames.add(canonicalName)
        }
    }

    private List recalculateOrderedClassTypes(List buildOrder) {
        def notOrderedClassTypes = index.values().collect { it.classType }.unique() - buildOrder
        if (notOrderedClassTypes) {
            println("WARNING [$name]: Not defined build order for class types: $notOrderedClassTypes")
        }
        buildOrder + notOrderedClassTypes
    }

    void printIndex() {
        printLine "Code index of $name"
        printLine()
        printItems collectClassesToGenerate()
        printItems collectMappers()
        dependsOn.each { it.printIndex() }
    }

    private List<String> collectClassesToGenerate() {
        index.findAll { it.value.toGenerate }
                .collect {
                    "$it.value.clazz.canonicalName ($it.key.classType for $it.key.model)".toString()
                }
                .sort()
    }

    private List<String> collectMappers() {
        mappersIndex.collect {
            "$it.key: $it.value".toString()
        }.sort()
    }

    private static void printItems(List items) {
        items.each { printLine "\t$it" }
        printLine()
    }

    private static void printLine(String message = null) {
        println message ?: '----------------------------------------------'
    }

}
