/*
 * Copyright (c) 2018,2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.model.java

import groovy.transform.AutoClone
import groovy.transform.EqualsAndHashCode
import groovy.transform.ToString
import groovy.transform.builder.Builder
import groovy.transform.builder.SimpleStrategy

import static pl.metaprogramming.codemodel.model.java.JavaDefs.MODIFIER_PRIVATE

@ToString(includePackage = false, includes = ['name', 'type'])
@EqualsAndHashCode(includes = ['name'])
@Builder(builderStrategy = SimpleStrategy)
@AutoClone
class FieldCm {
    def model
    String name
    String description
    ClassCd type
    List<AnnotationCm> annotations = []
    ValueCm value
    String modifiers = MODIFIER_PRIVATE
    boolean nonnull

    FieldCm() {}

    FieldCm(ClassCd type, String name = null) {
        this.type = type
        this.name = name
    }

    FieldCm(String type, String name) {
        this(new ClassCd(type), name)
    }

    void collectDependencies(Collection<ClassCd> dependencies) {
        type.collectDependencies(dependencies)
        annotations.each { it.collectDependencies(dependencies) }
    }

    boolean isStatic() {
        modifiers.contains('static')
    }

    boolean isFinal() {
        modifiers.contains('final')
    }

    FieldCm setFinal() {
        if (!modifiers.contains('final')) {
            modifiers = modifiers + ' final'
        }
        this
    }

    String getExpression() {
        value.toString()
    }

    FieldCm assign(String value) {
        this.value = ValueCm.value(value)
        this
    }

    FieldCm assign(ValueCm value) {
        this.value = value
        this
    }

    FieldCm alias(String newName) {
        new FieldCm(type, newName)
    }
}
