/*
 * Copyright (c) 2018,2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.model.java

import groovy.transform.builder.Builder
import groovy.transform.builder.SimpleStrategy

import static pl.metaprogramming.codemodel.model.java.JavaDefs.T_LIST

@Builder(builderStrategy = SimpleStrategy)
class ClassCd {
    static ClassCd VOID = new ClassCd('java.lang.Void')

    String packageName
    String className
    List<ClassCd> genericParams = []
    ClassCd extend
    boolean isInterface = false
    boolean isGeneric = false
    boolean isEnum = false
    boolean isArray = false

    static ClassCd genericParam(String param) {
        new ClassCd(className: param, isGeneric: true)
    }

    ClassCd() {}

    ClassCd(ClassCd prototype, List<ClassCd> genericParams = []) {
        this.packageName = prototype.packageName
        this.className = prototype.className
        this.genericParams = genericParams
    }

    ClassCd(String canonicalName, List<ClassCd> genericParams = []) {
        def name = parseCanonicalName(canonicalName)
        this.packageName = name[0]
        this.className = name[1]
        this.genericParams = genericParams
    }

    ClassCd(String packageName, String className, List<ClassCd> genericParams = []) {
        this.packageName = packageName
        this.className = className
        this.genericParams = genericParams
    }

    ClassCd withGeneric(ClassCd...genericParams) {
        new ClassCd(packageName, className, Arrays.asList(genericParams))
    }

    ClassCd array() {
        new ClassCd(this).setIsArray(true)
    }

    ClassCd list() {
        new ClassCd(T_LIST, [this])
    }

    FieldCm field(String name) {
        new FieldCm(this, name)
    }

    static private String[] parseCanonicalName(String canonicalName) {
        String[] result = new String[2]
        int lastDot = canonicalName.lastIndexOf('.')
        if (lastDot > 0) {
            result[0] = (canonicalName.substring(0, lastDot))
            result[1] = (canonicalName.substring(lastDot+1, canonicalName.length()))
        } else {
            result[1] = canonicalName
        }
        result
    }

    boolean isClass(String canonicalName) {
        def name = parseCanonicalName(canonicalName)
        name[0] == packageName && name[1] == className
    }

    boolean isClass(ClassCd classCd) {
        this.packageName == classCd.packageName && this.className == classCd.className
    }

    String getCanonicalName() {
        packageName ? "${packageName}.${className}" : className
    }

    void collectDependencies(Collection<ClassCd> dependencies) {
        dependencies.add(this)
        genericParams?.each { it.collectDependencies(dependencies) }
    }

    boolean isVoid() {
        this == VOID
    }

    @Override
    String toString() {
        if (isVoid()) {
            'void'
        } else if (isArray) {
            "$canonicalName[]"
        } else if (genericParams) {
            "$canonicalName<${genericParams.collect { it.toString() }.join(', ')}>"
        } else {
            canonicalName
        }
    }

    @Override
    boolean equals(Object obj) {
        obj instanceof ClassCd && obj.canonicalName == canonicalName
    }

    @Override
    int hashCode() {
        canonicalName.hashCode()
    }
}
