/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.formatter

import pl.metaprogramming.codemodel.model.java.*

class JavaCodeFormatter extends BaseJavaCodeFormatter implements CodeFormatter<ClassCm> {

    static final String JAVA_LANG_PKG = 'java.lang'

    ClassCm classCm

    String format(ClassCm codeModel) {
        format()
    }

    JavaCodeFormatter(ClassCm classCm) {
        this.classCm = classCm
        initImports()
    }

    void initImports() {
        def excludePackages = [classCm.packageName, JAVA_LANG_PKG] +
                classCm.imports.
                        findAll { !it.startsWith('static ') && it.endsWith('.*') }.
                        collect { it.substring(0, it.length() - 2) }
        Set<ClassCd> classes = []
        classCm.annotations.each { it.collectDependencies(classes) }
        classCm.extend?.collectDependencies(classes)
        classCm.interfaces.each { it.collectDependencies(classes) }
        classCm.fields.each { it.collectDependencies(classes) }
        classCm.enumItems.each { it.collectDependencies(classes) }
        classCm.methods.each { it.collectDependencies(classes) }
        List<String> result = []
        result.addAll(classCm.imports.findAll {
            it.endsWith('.*') || !matchPackage(it, excludePackages)
        })
        result.addAll(classes
                .findAll { it.packageName && !excludePackages.contains(it.packageName) && !it.isGeneric }
                .collect { "${it.packageName}.${it.className}".toString() })
        imports = result.unique().sort()
    }

    private boolean matchPackage(String importExp, List<String> packages) {
        packages.contains(importExp.substring(0, importExp.lastIndexOf('.')))
    }

    String format() {
        init(classCm.packageName)

        buf.add("package $classCm.packageName;").newLine()
        buf.addLines(imports.collect { "import $it;" }).newLine()

        if (classCm.description) {
            buf.addLines(formatJavadoc(classCm.description))
        }
        buf.addLines(formatAnnotations(classCm.annotations)).newLine()

        buf.add(formatClassHeader()).add(' {')
        buf.indent(1).newLine()

        addEnumItems(buf)
        addFields(buf)

        buf.indent().newLine()
        if (classCm.fields && classCm.methods) buf.newLine()
        addMethods(buf)

        buf.indent().newLine('}').newLine()
        buf.take()
    }

    private void addEnumItems(CodeBuffer buf) {
        EnumItemCm prevItem
        classCm.enumItems.each {
            if (prevItem) {
                buf.add(',')
                if (prevItem.description) buf.add(" // $prevItem.description")
            }
            buf.addLines(formatAnnotations(it.annotations))
            buf.newLine("${it.name}")
            if (it.value) {
                buf.add("(${it.value})")
            }
            prevItem = it
        }
        if (prevItem) {
            buf.add(';')
            if (prevItem.description) buf.add(" // $prevItem.description")
            buf.newLine()
        }
    }

    private void addFields(CodeBuffer buf) {
        def valueFields = classCm.fields.findAll { it.isFinal() && it.value != null }
        if (valueFields) {
            valueFields.each {
                addField(it, buf)
            }
            buf.newLine()
        }
        (classCm.fields - valueFields).each {
            addField(it, buf)
        }
    }

    private void addField(FieldCm field, CodeBuffer buf) {
        if (field.description) {
            buf.newLine().addLines(formatJavadoc(field.description))
        }
        buf.newLine().add(formatField(field))
        buf.add(';')
    }

    String formatField(FieldCm field) {
        formatAnnotations(field.annotations, SPACE) +
                field.modifiers + SPACE +
                formatTypeName(field) +
                (field.value == null ? '' : ' = ' + field.value)
    }

    private void addMethods(CodeBuffer buf) {
        classCm.methods.each {
            buf.indent(1)
            if (it.description) {
                buf.addLines(formatJavadoc(it.description))
            }
            buf.addLines(formatAnnotations(it.annotations))
            if (classCm.isInterface && !it.implBody) {
                buf.newLine(formatSignature(it)).add(";").newLine()
            } else {
                buf.newLine("${formatSignature(it)} {")
                buf.indent(2).addLines(it.implBody)
                buf.indent(1).newLine("}").indent().newLine()
            }
        }
    }

    String formatClassHeader() {
        def buf = new StringBuffer()
        buf.append(classCm.modifiers).append(' ')
        buf.append(classCm.isInterface ? 'interface' : classCm.isEnum ? 'enum' : 'class')
        buf.append(' ').append(classCm.className)
        buf.append(formatGenericParams(classCm.genericParams))
        if (classCm.extend) {
            buf.append(" extends ${formatUsage(classCm.extend)}")
        }
        if (classCm.interfaces.size() > 0) {
            buf.append(classCm.isInterface ? ' extends ' : ' implements ')
            buf.append(classCm.interfaces.collect { formatUsage(it) }.join(', '))
        }
        buf.toString()
    }


    String formatParam(FieldCm fieldCm) {
        "${formatAnnotations(fieldCm.annotations, SPACE)}${formatTypeName(fieldCm)}"
    }

    String formatTypeName(FieldCm fieldCm) {
        "${formatUsage(fieldCm.type)} ${fieldCm.name}"
    }

    static List<String> formatJavadoc(String comment) {
        def result = ['/**']
        comment.eachLine { result.add(' * ' + it) }
        result.add(' */')
        result
    }


    String formatSignature(MethodCm methodCm) {
        new ConcatenationBuilder()
                .append(formatModifiers(methodCm))
                .append(genericParams(methodCm))
                .append(methodCm.constructor ? null : formatUsage(methodCm.resultType))
                .append("${methodCm.name}(${methodCm.params ? methodCm.params.collect { formatParam(it) }.join(', ') : ''})")
                .toString()
    }

    String formatModifiers(MethodCm methodCm) {
        methodCm.ownerClass.isInterface
                ? methodCm.modifiers.replace('public ', '').replace('public', '')
                : methodCm.modifiers
    }

    String genericParams(MethodCm methodCm) {
        Set<ClassCd> genericParams = []
        if (methodCm.resultType) {
            collectGenericParams(methodCm.resultType, genericParams)
        }
        methodCm.params?.each { collectGenericParams(it.type, genericParams) }
        if (!methodCm.static) {
            genericParams.removeAll(methodCm.ownerClass.genericParams)
        }
        formatGenericParams(genericParams)
    }

    static collectGenericParams(ClassCd cm, Set<ClassCd> result) {
        if (cm.isGeneric && cm.className != '?') {
            result.add(cm)
        } else {
            cm.genericParams?.each { collectGenericParams(it, result) }
        }
    }

    static String toCapitalizeJavaName(String name) {
        toJavaName(name, true)
    }

    static String toJavaName(String name, boolean capitalize = false) {
        StringBuilder buf = new StringBuilder()
        boolean upperCase = false
        for (int i = 0; i < name.length(); i++) {
            char c = name.charAt(i)
            if (Character.isLetterOrDigit(c)) {
                if (buf.length() == 0 && Character.isDigit(c)) {
                    buf.append('_')
                }
                buf.append(buf.length() == 0 ? (capitalize ? Character.toUpperCase(c) : Character.toLowerCase(c))
                        : upperCase ? Character.toUpperCase(c) : c)
                upperCase = false
            } else {
                upperCase = true
            }
        }
        buf.toString()
    }


    static String toUpperCase(String javaName) {
        StringBuilder buf = new StringBuilder()
        for (int i = 0; i < javaName.length(); i++) {
            char c = javaName.charAt(i)
            if (Character.isUpperCase(c) && i > 0) {
                buf.append('_')
            }
            buf.append(Character.toUpperCase(c))
        }
        buf.toString()
    }

    static class ConcatenationBuilder {
        private StringBuilder builder = new StringBuilder()

        ConcatenationBuilder append(String text) {
            if (text?.trim()) {
                if (builder.size() > 0) {
                    builder.append(' ')
                }
                builder.append(text)
            }
            this
        }

        String toString() {
            builder.toString()
        }
    }

}
