/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.xml


import pl.metaprogramming.codemodel.builder.java.dto.BaseDtoBuildStrategy
import pl.metaprogramming.codemodel.model.java.AnnotationCm
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.ValueCm
import pl.metaprogramming.metamodel.data.DataSchema
import pl.metaprogramming.metamodel.data.ObjectType

import static pl.metaprogramming.codemodel.builder.java.ClassType.*
import static pl.metaprogramming.codemodel.builder.java.MetaModelAttribute.JAVA_NAME
import static pl.metaprogramming.codemodel.model.java.JavaDefs.*

class XmlDtoBuildStrategy extends BaseDtoBuildStrategy {

    static AnnotationCm XML_ACCESSOR_TYPE_ANNOTATION = new AnnotationCm('javax.xml.bind.annotation.XmlAccessorType', [value: ValueCm.value("XmlAccessType.FIELD")])

    @Override
    void makeImplementation() {
        super.makeImplementation()
        interfaces.clear()  // TODO don't add Serializable interface in BaseDtoBuildStrategy
        addImports("javax.xml.bind.annotation.XmlAccessType")
        addAnnotation(XML_ACCESSOR_TYPE_ANNOTATION)
        addAnnotation(prepareXmlTypeAnnotation(model))
        if (model.isRootElement) {
            addAnnotation(prepareXmlRootElementAnnotation(model))
        }
    }

    @Override
    void addField(DataSchema schema) {
        def classCd = getClass(schema.enumOrItemEnum ? XML_ENUM : XML_DTO, schema.dataType)
        addFields new FieldCm(
                name: schema.getAdditive(JAVA_NAME),
                description: schema.description,
                annotations: getFieldAnnotations(schema, classCd),
                type: classCd)
    }

    private List<AnnotationCm> getFieldAnnotations(DataSchema schema, ClassCd classCd) {
        def result = []
        result.add(schema.isRequired && !schema.isNillable ? ANNOT_NON_NULL : ANNOT_NULLABLE)
        addXmlElementAnnotation(result, schema)
        if (classCd == T_LOCAL_DATE) {
            addXmlJavaTypeAdapterAnnotation(result, JAXB_LOCAL_DATE_ADAPTER)
        }
        if (classCd == T_LOCAL_DATE_TIME) {
            addXmlJavaTypeAdapterAnnotation(result, JAXB_LOCAL_DATE_TIME_ADAPTER)
        }
        result
    }

    static void addXmlElementAnnotation(List<AnnotationCm> result, DataSchema schema) {
        Map<String, ValueCm> xmlElementAttributes = [:]
        if (schema.isRequired) {
            xmlElementAttributes.put('required', ValueCm.value('true'))
        }
        if (schema.isNillable) {
            xmlElementAttributes.put('nillable', ValueCm.value('true'))
        }
        if (schema.getAdditive(JAVA_NAME) != schema.code) {
            xmlElementAttributes.put('name', ValueCm.escaped(schema.code))
        }
        if (xmlElementAttributes) {
            result.add(new AnnotationCm('javax.xml.bind.annotation.XmlElement', xmlElementAttributes))
        }
    }

    private void addXmlJavaTypeAdapterAnnotation(List<AnnotationCm> result, def adapterClassType) {
        def adapterClass = getClass(adapterClassType)
        addImport(adapterClass)
        result.add(new AnnotationCm(
                'javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter',
                [value: ValueCm.value("${adapterClass.className}.class")]))
    }

    static AnnotationCm prepareXmlTypeAnnotation(ObjectType model) {
        new AnnotationCm('javax.xml.bind.annotation.XmlType', [
                name: ValueCm.escaped(model.isRootElement ? '' : model.code),
                propOrder: new ValueCm(
                        value: model.fields.collect { it.getAdditive(JAVA_NAME) },
                        array: true,
                        escaped: true,
                        itemPerLine: true)
        ])
    }

    static AnnotationCm prepareXmlRootElementAnnotation(ObjectType model) {
        new AnnotationCm('javax.xml.bind.annotation.XmlRootElement', [
                name: ValueCm.escaped(model.code)])
    }
}
