/*
 * Copyright (c) 2021 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.spring.mapper

import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.MethodCm
import pl.metaprogramming.metamodel.oas.HttpResponse

import static pl.metaprogramming.codemodel.builder.java.ClassType.DTO
import static pl.metaprogramming.codemodel.builder.java.ClassType.REST_DTO
import static pl.metaprogramming.codemodel.builder.java.spring.SpringDefs.HTTP_STATUS_CODE_EXCEPTION
import static pl.metaprogramming.codemodel.model.java.JavaDefs.ANNOT_SLF4J

class RestOutResponseFailMapperBuilder extends RestOutResponseMapperBuilder {

    @Override
    MethodCm makeDeclaration() {
        new MethodCm(
                name: 'map',
                resultType: responseClass,
                params: [new FieldCm(name: 'e', type: HTTP_STATUS_CODE_EXCEPTION)]
        )
    }

    @Override
    String makeImplBody() {
        def anyResponseWithContent = model.responses.any { !it.successResponse && it.schema }
        if (anyResponseWithContent) {
            strategy.addAnnotation(ANNOT_SLF4J)
            strategy.addImports('com.fasterxml.jackson.core.JsonProcessingException')
            codeBuf.tryBlock()
        }
        def nonDefaultFailResponses = model.responses.findAll { !it.default && !it.successResponse }
        nonDefaultFailResponses.each {
            codeBuf.ifBlock("e.getRawStatusCode() == $it.status", handle(it))
                    .endBlock()
        }
        if (model.defaultResponse) {
            codeBuf.addLines(handle(model.defaultResponse))
        } else {
            codeBuf.addLines("throw e;")
        }
        if (anyResponseWithContent) {
            codeBuf
                    .catchBlock("JsonProcessingException e2",
                            "log.error(\"Can't deserialize data\", e2);",
                            "throw e;")
                    .endBlock()
        }
        codeBuf.take()
    }

    private String handle(HttpResponse response) {
        String responseSetter = 'set' + (response.default ? 'Other' : response.status)
        def args = []
        if (response.default) {
            args.add('e.getRawStatusCode()')
        }
        if (response.schema) {
            def responseObject = response.schema.dataType
            def restDtoClass = strategy.getClass(REST_DTO, responseObject)
            strategy.addImports('com.fasterxml.jackson.databind.ObjectMapper', restDtoClass)
            String restDtoExpField = strategy.addVarDeclaration("restValue", restDtoClass, "new ObjectMapper().readValue(e.getResponseBodyAsString(), ${restDtoClass.className}.class)")
            args.add(strategy.transform(REST_DTO, DTO, responseObject, restDtoExpField).value)
        }
        "return ${responseObjectCreatePrefix}.${responseSetter}(${args.join(', ')});"
    }
}
