/*
 * Copyright (c) 2021 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.spring.mapper

import pl.metaprogramming.codemodel.builder.java.MetaModelAttribute
import pl.metaprogramming.codemodel.builder.java.mapper.BaseMethodCmBuilder
import pl.metaprogramming.codemodel.builder.java.spring.RestClientBuildHelper
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.MethodCm
import pl.metaprogramming.metamodel.oas.Operation
import pl.metaprogramming.metamodel.oas.Parameter
import pl.metaprogramming.metamodel.oas.ParamLocation

import static pl.metaprogramming.codemodel.builder.java.ClassType.*
import static pl.metaprogramming.codemodel.builder.java.spring.SpringDefs.HTTP_HEADERS
import static pl.metaprogramming.codemodel.builder.java.spring.SpringDefs.MULTI_VALUE_MAP
import static pl.metaprogramming.codemodel.model.java.JavaDefs.*

class RestOutRequestMapperBuilder extends BaseMethodCmBuilder<Operation> {

    private static final String REQUEST_PARAM = 'request'
    private static final String RAW_REQUEST_VAR = 'rawRequest'
    private static final String URI_VAR = 'uri'

    @Lazy
    protected RestClientBuildHelper helper = new RestClientBuildHelper(strategy, model)

    @Override
    MethodCm makeDeclaration() {
        new MethodCm(
                name: 'map',
                resultType: helper.requestEntity,
                params: [new FieldCm(name: REQUEST_PARAM, type: new ClassCd(strategy.getClass(REQUEST_DTO)), annotations: [ANNOT_NON_NULL])],
        )
    }

    @Override
    String makeImplBody() {
        strategy.addImports(T_URI,
                'org.springframework.web.util.UriComponentsBuilder')
        makeRawRequestVar()
        makeUriVar()
        def headersVar = makeHeadersVar()
        def bodyVar = makeBodyVar()
        codeBuf.addLines("return RequestEntity.${model.type.name().toLowerCase()}($URI_VAR)")
        codeBuf.indentInc(2)
        if (headersVar) {
            codeBuf.addLines(".headers($headersVar)")
        }
        if (hasRequestBody()) {
            codeBuf.addLines(".body($bodyVar);")
        } else {
            codeBuf.addLines('.build();')
        }
        codeBuf.indentInc(-2)
        codeBuf.take()
    }

    private void makeRawRequestVar() {
        def requestVarClass = strategy.getClass(REST_REQUEST_DTO)
        def mapper = strategy.findMapper(requestVarClass, methodCm.params.collect { it.type })
        strategy.addVarDeclaration(RAW_REQUEST_VAR, requestVarClass, "${mapper.name}($REQUEST_PARAM)")
    }

    private void makeUriVar() {
        def pathParams = makePathParamsVar()
        codeBuf
                .addLines("URI $URI_VAR = UriComponentsBuilder")
                .indentInc(2)
                .addLines(".fromUriString(${strategy.callComponent(ENDPOINT_PROVIDER, 'getEndpoint("' + model.path + '")')})")
                .addLines(model.getParameters(ParamLocation.QUERY).collect {
                    ".queryParam${paramNameValue(it)}"
                })
        if (pathParams) {
            codeBuf.addLines(".build($pathParams);")
        } else {
            codeBuf.addLines('.build()', '.toUri();')
        }
        codeBuf.indentInc(-2)
    }

    private String makePathParamsVar() {
        def pathParams = model.getParameters(ParamLocation.PATH)
        if (pathParams.empty) {
            return ''
        }
        strategy.addImports(T_MAP, T_HASH_MAP)
        codeBuf.addLines("Map<String, String> pathParams = new HashMap<>();")
        codeBuf.addLines(pathParams.collect {
            "pathParams.put${paramNameValue(it)};"
        })
        "pathParams"
    }

    private String paramNameValue(Parameter parameter) {
        "(\"$parameter.name\", ${strategy.getter([RAW_REQUEST_VAR, parameter.name])})"
    }

    private String makeBodyVar() {
        model.multipart ? makeMultipartBodyVar()
                : model.requestBodySchema ? strategy.getter([RAW_REQUEST_VAR, model.requestBodySchema.code]) : null
    }

    private String makeMultipartBodyVar() {
        strategy.addImports(MULTI_VALUE_MAP,
                'org.springframework.util.LinkedMultiValueMap')
        String var = 'body'
        codeBuf.addLines("MultiValueMap<String, Object> $var = new LinkedMultiValueMap<>();")
        codeBuf.addLines(model.getParameters(ParamLocation.FORMDATA).collect {
            "${var}.add${paramNameValue(it)};"
        })
        codeBuf.addLines(model.multipartFormDataRequestBody?.fields?.collect {
            "${var}.add(\"$it.code\", ${strategy.getter([RAW_REQUEST_VAR, it.code])});"
        })
        var
    }

    private String makeHeadersVar() {
        def headers = model.getParameters(ParamLocation.HEADER)
        if (headers.empty) {
            return null
        }
        String var = 'headers'
        strategy.addImports(OPTIONAL)
        addVarDeclaration(var, HTTP_HEADERS, 'new')
        codeBuf.addLines(headers.collect {
            "Optional.ofNullable(${paramGetter(RAW_REQUEST_VAR, it)}).ifPresent(v -> ${var}.add(\"$it.name\", v));"
        })
        var
    }

    private String paramGetter(String varName, Parameter param) {
        "${varName}.get${(param.additives[MetaModelAttribute.JAVA_NAME] as String).capitalize()}()"
    }

    private boolean hasRequestBody() {
        model.requestBodySchema || model.multipart
    }
}
