/*
 * Copyright (c) 2021 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.spring.mapper

import pl.metaprogramming.codemodel.builder.java.mapper.BaseMethodCmBuilder
import pl.metaprogramming.codemodel.builder.java.spring.RestClientBuildHelper
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.MethodCm
import pl.metaprogramming.metamodel.oas.Operation

import static pl.metaprogramming.codemodel.builder.java.ClassType.REQUEST_DTO
import static pl.metaprogramming.codemodel.builder.java.ClassType.RESPONSE_DTO
import static pl.metaprogramming.codemodel.builder.java.spring.SpringDefs.HTTP_STATUS_CODE_EXCEPTION
import static pl.metaprogramming.codemodel.builder.java.spring.SpringDefs.REST_TEMPLATE
import static pl.metaprogramming.codemodel.model.java.JavaDefs.ANNOT_NON_NULL

class RestOutInvokeMethodBuilder extends BaseMethodCmBuilder<Operation> {

    private static final String REQUEST_PARAM = 'request'
    private static final FieldCm REST_TEMPLATE_FIELD = new FieldCm(name: 'restTemplate', type: REST_TEMPLATE)

    private Operation operation
    @Lazy
    private RestClientBuildHelper helper = new RestClientBuildHelper(strategy, operation)

    RestOutInvokeMethodBuilder() {
        modelMapper = { operation }
    }

    @Override
    MethodCm makeDeclaration() {
        new MethodCm(
                name: operation.code,
                description: operation.description,
                resultType: getClass(RESPONSE_DTO),
                params: [new FieldCm(name: REQUEST_PARAM, type: getClass(REQUEST_DTO), annotations: [ANNOT_NON_NULL])],
        )
    }

    @Override
    String makeImplBody() {
        setup()
        codeBuf
                .reset()
                .tryBlock("return ${getSuccessResponseMapper()};")
                .catchBlock("HttpStatusCodeException e",
                        "return $failResponseMapper;"
                )
                .endBlock()
                .take()
    }

    private void setup() {
        if (strategy.findFields { it == REST_TEMPLATE_FIELD }.empty) {
            strategy.addFields(REST_TEMPLATE_FIELD)
        }
        strategy.addImports('org.springframework.web.client.HttpStatusCodeException')
    }

    private String getRestCall() {
        def responseClass = helper.responseClass
        strategy.addImports(responseClass)
        def exchangeArgs = [
                strategy.transform(methodCm.params, helper.requestEntity).value.toString(),
                getResponseClassType(responseClass)]
        "restTemplate.exchange(${exchangeArgs.join(', ')})"
    }

    private String getResponseClassType(ClassCd responseClass) {
        if (responseClass.genericParams) {
            def responseClassType = classNameFormatter.format(responseClass)
            strategy.addImports('org.springframework.core.ParameterizedTypeReference')
            "new ParameterizedTypeReference<$responseClassType>() {}"
        } else {
            classNameFormatter.classRef(responseClass)
        }
    }

    private String getSuccessResponseMapper() {
        strategy.transform(new FieldCm(helper.responseEntity).assign(restCall), getClass(RESPONSE_DTO)).value
    }

    private String getFailResponseMapper() {
        strategy.transform(new FieldCm(HTTP_STATUS_CODE_EXCEPTION).assign('e'), getClass(RESPONSE_DTO)).value
    }
}
