/*
 * Copyright (c) 2021 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.spring.mapper

import pl.metaprogramming.codemodel.builder.java.mapper.BaseMethodCmBuilder
import pl.metaprogramming.codemodel.builder.java.spring.SpringDefs
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.MethodCm
import pl.metaprogramming.metamodel.data.DataSchema
import pl.metaprogramming.metamodel.oas.HttpResponse
import pl.metaprogramming.metamodel.oas.Operation

import static pl.metaprogramming.codemodel.builder.java.ClassType.*
import static pl.metaprogramming.codemodel.model.java.JavaDefs.ANNOT_NON_NULL

class RestInResponseMapperBuilder extends BaseMethodCmBuilder<Operation> {

    @Override
    MethodCm makeDeclaration() {
        new MethodCm(
                name: 'map',
                resultType: SpringDefs.RESPONSE_ENTITY,
                params: [new FieldCm(name: 'response', type: new ClassCd(strategy.getClass(RESPONSE_DTO)), annotations: [ANNOT_NON_NULL])]
        )
    }

    @Override
    String makeImplBody() {
        codeBuf.addLines(
                'ResponseEntity.BodyBuilder responseBuilder = ResponseEntity.status(response.getStatus());',
                'response.getHeaders().forEach(responseBuilder::header);'
        )
        if (model.responses.size() == 1) {
            codeBuf.addLines("return ${prepareResponseMap(model.responses[0])};")
        } else {
            def responses = model.responses.sort { -it.status }
            responses.take(responses.size() - 1).each {
                codeBuf
                        .ifBlock("response.is${it.status}()",
                                "return ${prepareResponseMap(it)};")
                        .endBlock()
            }
            codeBuf.addLines("return ${prepareResponseMap(responses.last())};")
        }
        codeBuf.take()
    }

    private String prepareResponseMap(HttpResponse response) {
        if (response.schema == null) {
            'responseBuilder.build()'
        } else {
            String responseGetter = "response.get${response.status ?: 'Other'}()"
            "responseBuilder.body(${makeTransformation(response.schema, responseGetter)})"
        }
    }

    private String makeTransformation(DataSchema schema, String fieldName) {
        if (schema.dataType.isBinary()) {
            return fieldName
        }
        def javaClass = strategy.getClass(DTO, schema.dataType)
        def jsonClass = strategy.getClass(REST_DTO, schema.dataType)
        strategy.makeTransformation(jsonClass, [new FieldCm(javaClass).assign(fieldName)])
    }
}
