/*
 * Copyright (c) 2021 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.spring.mapper

import groovy.transform.builder.Builder
import groovy.transform.builder.SimpleStrategy
import pl.metaprogramming.codemodel.builder.java.mapper.BaseMethodCmBuilder
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.ClassCm
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.MethodCm
import pl.metaprogramming.metamodel.data.DataSchema
import pl.metaprogramming.metamodel.data.ObjectType

import static pl.metaprogramming.codemodel.model.java.JavaDefs.MODIFIER_PRIVATE
import static pl.metaprogramming.codemodel.model.java.JavaDefs.MODIFIER_PUBLIC

@Builder(builderStrategy = SimpleStrategy)
class DtoMapperBuilder extends BaseMethodCmBuilder {
    static final String PARAM_FIELD_NAME = 'value'

    ObjectType objectType
    ClassCd to
    ClassCd from
    List<FieldCm> fromFields
    boolean handleNullParam
    boolean isPrivate
    boolean resultAsParam
    boolean callMapperWithResultParamAsImpl
    boolean throwNotImplementedExceptionAsImpl

    @Override
    MethodCm makeDeclaration() {
        new MethodCm(
                name: makeName(),
                resultType: to,
                params: makeParams(),
                modifiers: isPrivate ? MODIFIER_PRIVATE : MODIFIER_PUBLIC,
        )
    }

    private String makeName() {
        resultAsParam ? 'map' : "map2$to.className"
    }

    private List<FieldCm> makeParams() {
        if (fromFields != null) {
            fromFields
        } else {
            List<FieldCm> params = [new FieldCm(from, PARAM_FIELD_NAME)]
            if (resultAsParam) {
                params.add(0, new FieldCm(to, 'result'))
            }
            params
        }
    }

    @Override
    String makeImplBody() {
        try {
            if (throwNotImplementedExceptionAsImpl) {
                'throw new RuntimeException("Not implemented");'
            } else if (callMapperWithResultParamAsImpl) {
                def mapper = strategy.findMapper(to, [to, from])
                "return $PARAM_FIELD_NAME == null ? null : ${mapper.name}(new ${to.className}(), $PARAM_FIELD_NAME);"
            } else {
                makeMapperImplBody()
            }
        } catch (Exception e) {
            throw new IllegalStateException("Can't make implementation for ${strategy.classModel.className}.${methodCm.name}", e)
        }
    }

    private String makeMapperImplBody() {
        if (resultAsParam) {
            handleSuperClass()
            codeBuf.add("return result")
        } else if (handleNullParam) {
            codeBuf.add("return value == null ? null : new ${to.className}()")
        } else {
            codeBuf.add("return new ${to.className}()")
        }
        codeBuf.indent(2)
        objectType.fields.each { addMapping(it) }
        codeBuf.newLine().add(';').take()
    }

    private void handleSuperClass() {
        if (to instanceof ClassCm && to.extend) {
            codeBuf.add(strategy.makeTransformation(to.extend, methodCm.params.collect {
                new FieldCm((it.type as ClassCm).extend, it.name)
            })).add(';').newLine()
        }
    }

    private FieldCm addMapping(DataSchema schema) {
        try {
            def toField = (to as ClassCm).getField(schema)
            def fromField = getFromField(schema)
            def transformation = strategy.makeTransformation(
                    toField.type, [transformParam(fromField, schema)])
            codeBuf.newLine().add(".set${toField.name.capitalize()}($transformation)")
            fromField
        } catch (Exception e) {
            throw new IllegalStateException("Can't generate field for $strategy.modelName : $schema", e)
        }
    }

    private FieldCm transformParam(FieldCm fromField, DataSchema schema) {
        if (fromFields) {
            if (schema.defaultValue) {
                strategy.addImports('java.util.Optional')
                return new FieldCm().setType(fromField.type)
                        .assign("Optional.ofNullable($fromField.name).orElse(${DefaultValueMapper.make(fromField.type, schema.defaultValue, strategy)})")
            }
            fromField
        } else {
            new FieldCm(fromField.type, "${PARAM_FIELD_NAME}.${fromField.name}")
        }
    }

    private FieldCm getFromField(Object schema) {
        if (fromFields) {
            fromFields.find { it.model == schema }
        } else {
            (from as ClassCm).getField(schema)
        }
    }

}
