/*
 * Copyright (c) 2021 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.spring.mapper

import pl.metaprogramming.codemodel.builder.java.ClassCmBuildStrategy
import pl.metaprogramming.codemodel.builder.java.base.BaseEnumBuildStrategy
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.ClassCm
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.ValueCm

import static pl.metaprogramming.codemodel.model.java.JavaDefs.*

class DefaultValueMapper {

    static ValueCm make(ClassCd type, String value, ClassCmBuildStrategy buildStrategy) {
        if (value == null) {
            ValueCm.value('null')
        } else if (type == T_STRING) {
            ValueCm.escaped(value)
        } else if ([T_INTEGER, T_SHORT].contains(type)) {
            ValueCm.value(value)
        } else if (type == T_LONG) {
            ValueCm.value(value + 'l')
        } else if (type == T_FLOAT) {
            ValueCm.value(value + 'f')
        } else if (type == T_DOUBLE) {
            ValueCm.value(value + (value.contains('.') ? '' : 'd'))
        } else if (type == T_BIG_DECIMAL) {
            ValueCm.value("new BigDecimal(\"${value}\")")
        } else if (type == T_ZONED_DATE_TIME) {
            ValueCm.value("ZonedDateTime.parse(\"${value}\")")
        } else if (type instanceof ClassCm && type.isEnum) {
            ValueCm.value("${type.className}.${BaseEnumBuildStrategy.toEnumItemName(value)}")
        } else {
            ValueCm.value(buildStrategy.makeTransformation(type, [new FieldCm(type: T_STRING, value: ValueCm.escaped(value))]))
        }
    }

}
