/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.spring


import pl.metaprogramming.codemodel.builder.java.ClassCmBuildStrategy
import pl.metaprogramming.codemodel.model.java.MethodCm
import pl.metaprogramming.metamodel.wsdl.WsdlApi

import static pl.metaprogramming.codemodel.builder.java.ClassType.ENDPOINT_PROVIDER
import static pl.metaprogramming.codemodel.builder.java.ClassType.WS_CLIENT

class WsClientConfigurationBuildStrategy extends ClassCmBuildStrategy<WsdlApi> {

    @Override
    void makeImplementation() {
        addAnnotation(SpringDefs.ANNOT_CONFIGURATION)
        addMethods(prepareCreateClientMethod())
    }

    private MethodCm prepareCreateClientMethod() {
        def clientClass = getClass(WS_CLIENT)
        new MethodCm(
                name: "create${clientClass.className}",
                resultType: clientClass,
                annotations: [SpringDefs.ANNOT_BEAN],
                implBody: prepareCreateClientMethodBody()
        )
    }

    private String prepareCreateClientMethodBody() {
        def packages = model.schemaPackages.collect { "\"$it\"" }.join(", ")
        def resultClass = getClass(WS_CLIENT).className

        addImports('org.springframework.oxm.jaxb.Jaxb2Marshaller')
        codeBuf
                .newLine('Jaxb2Marshaller marshaller = new Jaxb2Marshaller();')
                .newLine("marshaller.setContextPaths($packages);")
                .newLine("$resultClass client = new $resultClass();")
                .newLine("client.setDefaultUri(${getEndpointUri()});")
                .newLine("client.setMarshaller(marshaller);")
                .newLine("client.setUnmarshaller(marshaller);")
                .newLine("return client;")
                .take()
    }

    String getEndpointUri() {
        getClassOptional(ENDPOINT_PROVIDER)
                .map({
                    def endpointProviderField = injectDependency(it)
                    "${endpointProviderField.name}.getEndpoint(\"${model.uri.toURL().path}\")"
                }).orElseGet({ "\"${model.uri}\"" })
    }
}
