/*
 * Copyright (c) 2021 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.spring

import groovy.transform.builder.Builder
import groovy.transform.builder.SimpleStrategy
import pl.metaprogramming.codemodel.builder.java.ClassCmBuildStrategy
import pl.metaprogramming.codemodel.builder.java.config.SpringRestParams
import pl.metaprogramming.codemodel.model.java.*
import pl.metaprogramming.metamodel.data.DataSchema
import pl.metaprogramming.metamodel.data.DataTypeCode
import pl.metaprogramming.metamodel.oas.Operation
import pl.metaprogramming.metamodel.oas.ParamLocation
import pl.metaprogramming.metamodel.oas.Parameter

import java.util.function.Supplier

import static pl.metaprogramming.codemodel.formatter.JavaCodeFormatter.toJavaName
import static pl.metaprogramming.codemodel.model.java.JavaDefs.Validation.VALID

@Builder(builderStrategy = SimpleStrategy)
class RestParamsBuilder {

    static final AnnotationCm REQUEST_BODY_ANNOTATIONS = new AnnotationCm('org.springframework.web.bind.annotation.RequestBody')

    static final Map<ParamLocation, String> PARAM_ANNOTATIONS = [
            (ParamLocation.PATH)  : 'PathVariable',
            (ParamLocation.QUERY) : 'RequestParam',
            (ParamLocation.HEADER): 'RequestHeader',
            (ParamLocation.COOKIE): 'CookieValue',
    ]

    RestParamsBuilder(ClassCmBuildStrategy builder, Object bodyTypeOfCode) {
        this.builder = builder
        this.bodyTypeOfCode = bodyTypeOfCode
    }

    ClassCmBuildStrategy builder
    Operation operation
    boolean addControllerAnnotations
    boolean useJakartaBeanValidation
    boolean useDataAutoconversion;
    Object bodyTypeOfCode

    MethodCm makeControllerMethod(Operation operation) {
        this.operation = operation
        addControllerAnnotations = true
        new MethodCm(
                name: operation.code,
                description: operation.description,
                resultType: SpringDefs.RESPONSE_ENTITY,
                annotations: [makeMethodAnnotation()],
                params: make(operation),
        )
    }

    List<FieldCm> make(Operation operation) {
        this.operation = operation
        List<FieldCm> params = []
        if (builder.getParams(SpringRestParams).payloadField && operation.requestBodySchema != null) {
            params.add new FieldCm(
                    type: JavaDefs.T_STRING,
                    name: builder.getParams(SpringRestParams).payloadField,
                    annotations: annotations({ [REQUEST_BODY_ANNOTATIONS] })
            )
        }

        operation.requestSchema.fields.each { schema ->
            String javaName = toJavaName(schema.code)
            if (schema instanceof Parameter) {
                def param = schema as Parameter
                params.add new FieldCm(
                        name: javaName,
                        type: getParamType(schema),
                        model: schema,
                        annotations: makeParamAnnotations(param, javaName)
                )
            } else {
                params.add new FieldCm(
                        name: javaName,
                        type: builder.getClass(bodyTypeOfCode, operation.requestBodySchema.dataType),
                        model: schema,
                        annotations: makeBodyParamAnnotations()
                )
            }
        }
        params
    }


    private AnnotationCm makeMethodAnnotation() {
        Map<String, ValueCm> params = [
                value   : ValueCm.escaped(operation.path),
                produces: ValueCm.escapedArray(operation.produces),
        ]
        if (operation.type.hasBody()) {
            params.consumes = ValueCm.escapedArray(operation.consumes)
        }
        String canonicalName = "org.springframework.web.bind.annotation.${operation.type.name().toLowerCase().capitalize()}Mapping"
        new AnnotationCm(canonicalName, params)
    }


    private List<AnnotationCm> annotations(Supplier<List<AnnotationCm>> producer) {
        addControllerAnnotations ? producer.get() : []
    }

    protected ClassCd getParamType(DataSchema schema) {
        if (schema.isType(DataTypeCode.BINARY)) {
            SpringDefs.MULTIPART_FILE
        } else {
            builder.getClass(bodyTypeOfCode, schema.dataType)
        }
    }

    protected List<AnnotationCm> makeBodyParamAnnotations() {
        if (!addControllerAnnotations) {
            return []
        }
        def result = [REQUEST_BODY_ANNOTATIONS]
        if (useJakartaBeanValidation) {
            result.add(VALID)
        }
        result
    }

    protected List<AnnotationCm> makeParamAnnotations(Parameter param, String javaName) {
        if (!addControllerAnnotations) {
            return []
        }
        Map<String, ValueCm> params = [:]
        if (param.name != javaName) params.put('value', ValueCm.escaped(param.name))
        if (!(useDataAutoconversion && param.isRequired)) {
            params.put('required', ValueCm.value('false'))
        }
        String annotation = getParamAnnotation(param.location)
        def result = []
        result.add(new AnnotationCm("org.springframework.web.bind.annotation.$annotation" as String, params))
        if (useDataAutoconversion) {
            if (param.isTypeOrItemType(DataTypeCode.DATE_TIME)) {
                result.add(makeDateTimeFormatAnnotation('DATE_TIME'))
            } else if (param.isTypeOrItemType(DataTypeCode.DATE)) {
                result.add(makeDateTimeFormatAnnotation('DATE'))
            }
        }
        result
    }

    private AnnotationCm makeDateTimeFormatAnnotation(String isoFormat) {
        new AnnotationCm(
                'org.springframework.format.annotation.DateTimeFormat',
                ['iso': ValueCm.value('DateTimeFormat.ISO.' + isoFormat)]
        )
    }

    protected String getParamAnnotation(ParamLocation paramLocation) {
        if (paramLocation == ParamLocation.FORMDATA) {
            return 'RequestPart'
        }
        String annotation = PARAM_ANNOTATIONS.get(paramLocation)
        if (!annotation) {
            throw new RuntimeException("[$operation.code] Can't handle parameter location $paramLocation")
        }
        annotation
    }

}
