/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.spring

import pl.metaprogramming.codemodel.builder.java.ClassCmBuildStrategy
import pl.metaprogramming.codemodel.builder.java.config.SpringRestParams
import pl.metaprogramming.codemodel.model.java.*
import pl.metaprogramming.metamodel.data.DataSchema
import pl.metaprogramming.metamodel.data.DataType
import pl.metaprogramming.metamodel.oas.Operation
import pl.metaprogramming.metamodel.oas.ParamLocation

import static pl.metaprogramming.codemodel.builder.java.ClassType.REST_DTO
import static pl.metaprogramming.codemodel.formatter.JavaCodeFormatter.toJavaName

class RestControllerSignatureMethodBuilder {

    static final List<AnnotationCm> REQUEST_BODY_ANNOTATIONS = [new AnnotationCm('org.springframework.web.bind.annotation.RequestBody')]

    static final Map<ParamLocation, String> PARAM_ANNOTATIONS = [
            (ParamLocation.PATH)  : 'PathVariable',
            (ParamLocation.QUERY) : 'RequestParam',
            (ParamLocation.HEADER): 'RequestHeader',
            (ParamLocation.COOKIE): 'CookieValue',
    ]

    private Operation operation
    private ClassCmBuildStrategy builder

    RestControllerSignatureMethodBuilder(ClassCmBuildStrategy builder) {
        this.builder = builder
    }

    MethodCm make(Operation operation) {
        this.operation = operation
        prepareMethodCm()
    }

    MethodCm prepareMethodCm() {
        new MethodCm(
                name: operation.code,
                description: operation.description,
                resultType: SpringDefs.RESPONSE_ENTITY,
                annotations: [requestHandlerAnnotation()],
                params: requestHandlerParams(),
        )
    }

    protected AnnotationCm requestHandlerAnnotation() {
        Map<String, ValueCm> params = [
                value   : ValueCm.escaped(operation.path),
                produces: ValueCm.escapedArray(operation.produces),
        ]
        if (operation.type.hasBody()) {
            params.consumes = ValueCm.escapedArray(operation.consumes)
        }
        String canonicalName = "org.springframework.web.bind.annotation.${operation.type.name().toLowerCase().capitalize()}Mapping"
        new AnnotationCm(canonicalName, params)
    }

    protected List<FieldCm> requestHandlerParams() {
        List<FieldCm> params = []
        operation.parameters.each {
            String javaName = toJavaName(it.code ?: it.name)
            params.add(new FieldCm(
                    name: javaName,
                    type: getParamType(it),
                    annotations: [makeAnnotation(it.name, javaName, it.location)]))
        }
        if (operation.requestBodySchema != null) {
            if (builder.getParams(SpringRestParams).payloadField) {
                def payloadField = new FieldCm(JavaDefs.T_STRING, builder.getParams(SpringRestParams).payloadField)
                payloadField.annotations = REQUEST_BODY_ANNOTATIONS
                params.add(payloadField)
            }
            if (operation.multipartFormDataRequestBody) {
                operation.multipartFormDataRequestBody.fields.each {
                    params.add(new FieldCm(
                            name: toJavaName(it.code),
                            type: builder.getClass(REST_DTO, it.dataType),
                            annotations: [makeAnnotation(it.code, toJavaName(it.code), ParamLocation.FORMDATA)]))
                }
            } else {
                params.add(new FieldCm(
                        name: toJavaName(operation.requestBodySchema.code),
                        type: builder.getClass(REST_DTO, operation.requestBodySchema.dataType),
                        annotations: REQUEST_BODY_ANNOTATIONS))
            }
        }
        params
    }

    protected ClassCd getParamType(DataSchema dataSchema) {
        if (dataSchema.isArray()) {
            new ClassCd(JavaDefs.T_LIST, [getParamType(dataSchema.arrayType.itemsSchema)])
        } else if (dataSchema.dataType == DataType.BINARY) {
            SpringDefs.MULTIPART_FILE
        } else {
            JavaDefs.T_STRING
        }
    }

    protected AnnotationCm makeAnnotation(String paramName, String javaName, ParamLocation paramLocation) {
        Map<String, ValueCm> params = [:]
        if (paramName != javaName) params.put('value', ValueCm.escaped(paramName))
        params.put('required', ValueCm.value('false'))
        String annotation = getParamAnnotation(paramLocation)
        new AnnotationCm("org.springframework.web.bind.annotation.$annotation" as String, params)
    }

    protected String getParamAnnotation(ParamLocation paramLocation) {
        if (paramLocation == ParamLocation.FORMDATA) {
            return 'RequestPart'
        }
        String annotation = PARAM_ANNOTATIONS.get(paramLocation)
        if (!annotation) {
            throw new RuntimeException("[$operation.code] Can't handle parameter location $paramLocation")
        }
        annotation
    }
}
