/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.spring

import pl.metaprogramming.codemodel.builder.java.ClassCmBuildStrategy
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.MethodCm
import pl.metaprogramming.metamodel.oas.Operation

import static pl.metaprogramming.codemodel.builder.java.ClassType.*
import static pl.metaprogramming.codemodel.builder.java.spring.SpringDefs.*

class RestControllerMoBuildStrategy extends ClassCmBuildStrategy<List<Operation>> {

    @Override
    void makeImplementation() {
        addAnnotation(ANNOT_REST_CONTROLLER)
        addImports(REST_CONTROLLER_IMPORTS)

        def methodBuilder = new RestControllerMethodBuilder()

        model.each {
            addMethods(methodBuilder.make(it))
        }
    }


    class RestControllerMethodBuilder {

        RestParamsBuilder signatureBuilder
        Operation operation
        MethodCm methodCm

        RestControllerMethodBuilder() {
            signatureBuilder = new RestParamsBuilder(RestControllerMoBuildStrategy.this, REST_DTO)
        }

        MethodCm make(Operation operation) {
            this.operation = operation
            methodCm = signatureBuilder.makeControllerMethod(operation)
            methodCm.implBody = prepareImplBody()
            methodCm
        }

        String prepareImplBody() {
            def restRequestHandler = getClass(REST_REQUEST_HANDLER_TEMPLATE)
            methodCm.implDependencies = [restRequestHandler]
            codeBuf
                    .addLines("return ${restRequestHandler.className}.handle(")
                    .indent(2)
                    .addLines("${makeTransformation(getClass(REST_REQUEST_DTO), methodCm.params)},",
                            "${transformWithMethodRef(REST_REQUEST_DTO, VALIDATION_RESULT)},",
                            "${transformWithMethodRef(VALIDATION_RESULT, RESPONSE_ENTITY)},",
                            "${transformWithMethodRef(REST_REQUEST_DTO, REQUEST_DTO)},",
                            "${transformWithMethodRef(REQUEST_DTO, RESPONSE_DTO)},",
                            "${transformWithMethodRef(RESPONSE_DTO, RESPONSE_ENTITY)});")
                    .take()
        }

        String transformWithMethodRef(Object to, Object from) {
            RestControllerMoBuildStrategy.this.transformWithMethodRef(
                    to instanceof ClassCd ? to : getClass(to),
                    from instanceof ClassCd ? from : getClass(from)
            )
        }

        ClassCd getClass(def classType) {
            getClass(classType, operation)
        }
    }
}
