/*
 * Copyright (c) 2020 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.rest

import groovy.transform.builder.Builder

@Builder
class ValidatorPointer {

    static final String FIELD_PTR = 'field:'
    static final String DI_BEAN_PTR = 'di-bean:'
    static final String VALIDATION_BEAN_PTR = 'bean:'

    String classCanonicalName
    String staticField
    String method
    boolean isDiBean
    boolean isValidationBean

    /**
     * Expression should have the format:
     *  - <di-bean>:<class-canonical-name>[:method]
     *  - <di-bean>:<class-canonical-name>.method(args)
     *  - field:<class-canonical-name>.<static-field-name>
     * where class-canonical-name could be a 'VALIDATION_COMMON_CHECKERS'
     * which point to generated VALIDATION_COMMON_CHECKERS class.
     *
     * @param expression
     * @return
     */
    static ValidatorPointer fromExpression(String expression) {
        if (expression.startsWith(FIELD_PTR)) {
            String val = expression.substring(FIELD_PTR.length())
            def idx = val.lastIndexOf('.')
            builder().classCanonicalName(val.substring(0, idx))
                    .staticField(val.substring(idx + 1))
                    .build()
        } else if (expression.startsWith(DI_BEAN_PTR)) {
            String[] val = expression.substring((DI_BEAN_PTR.length())).split(':')
            builder().classCanonicalName(val[0])
                    .method(val.length > 1 ? val[1] : null)
                    .isDiBean(true)
                    .build()
        } else if (expression.startsWith(VALIDATION_BEAN_PTR)) {
            String[] val = expression.substring((VALIDATION_BEAN_PTR.length())).split(':')
            builder().classCanonicalName(val[0])
                    .method(val.length > 1 ? val[1] : null)
                    .isValidationBean(true)
                    .build()
        } else {
            throw new RuntimeException("Invalid expression: $expression")
        }
    }

    static boolean isValidatorPointerExpression(String expression) {
        [FIELD_PTR, DI_BEAN_PTR, VALIDATION_BEAN_PTR].any { expression.startsWith(it) }
    }
}
