/*
 * Copyright (c) 2021 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.rest

import groovy.transform.TypeChecked
import groovy.transform.TypeCheckingMode
import pl.metaprogramming.codemodel.builder.java.ClassCmBuildStrategy
import pl.metaprogramming.codemodel.builder.java.ClassType
import pl.metaprogramming.codemodel.builder.java.config.ValidationParams
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.MethodCm
import pl.metaprogramming.codemodel.model.java.ValueCm

import static pl.metaprogramming.codemodel.model.java.JavaDefs.*

class ValidationResultBuildStrategy extends ClassCmBuildStrategy<Object> {

    @Override
    void makeImplementation() {
        addAnnotation ANNOT_LB_GETTER
        addInterfaces I_SERIALIZABLE
        addField('stopped', T_BOOLEAN_PRIMITIVE)
        def requestField = addField('request', T_OBJECT)
        addField('status', T_INTEGER)

        addImports('java.util.ArrayList')
        addField('errors', errorClass.list()).setValue(ValueCm.value('new ArrayList<>()'))

        addMethods(
                new MethodCm(
                        name: classModel.className,
                        params: [requestField],
                        implBody: "this.request = request;"
                ),
                new MethodCm(
                        name: 'isValid',
                        resultType: T_BOOLEAN_PRIMITIVE,
                        implBody: 'return status == null;'
                ),
                new MethodCm(
                        name: 'isFieldValid',
                        resultType: T_BOOLEAN_PRIMITIVE,
                        params: [new FieldCm(T_STRING, 'field')],
                        implBody: codeBuf.block(
                                "for (ValidationError e : errors)",
                                "if (field != null && field.equals(e.getField())) return false;").endBlock()
                                .addLines('return true;').take()
                ),
                new MethodCm(
                        name: 'setStatus',
                        params: [T_INT_PRIMITIVE.field('status')],
                        implBody: 'this.status = status;'
                ),
                new MethodCm(
                        name: 'addError',
                        params: [errorClass.field('error')],
                        resultType: classModel,
                        implBody: codeBuf.addLines(
                                'status = error.getStatus() == null ? 400 : error.getStatus();',
                                'errors.add(error);',
                                shouldStopAfterFirstError()
                                        ? 'stopped = true;'
                                        : 'stopped = error.isStopValidation();',
                                'return this;'
                        ).take()

                ),
                new MethodCm(
                        name: 'getMessage',
                        resultType: T_STRING,
                        implBody: 'return errors.get(0).getMessage();'
                ),
                new MethodCm(
                        name: 'setError',
                        params: [T_INT_PRIMITIVE.field('status'), T_STRING.field('message')],
                        implBody: 'addError(ValidationError.builder().status(status).message(message).stopValidation(true).build());'
                ),
        )
    }

    ClassCd getErrorClass() {
        getClass(ClassType.VALIDATION_ERROR)
    }

    @TypeChecked(TypeCheckingMode.SKIP)
    boolean shouldStopAfterFirstError() {
        getParams(ValidationParams).stopAfterFirstError
    }

}
