/*
 * Copyright (c) 2020 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.rest

import groovy.transform.TypeChecked
import groovy.transform.TypeCheckingMode
import pl.metaprogramming.codemodel.builder.java.ClassCmBuildStrategy
import pl.metaprogramming.codemodel.builder.java.config.SpringRestParams
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.MethodCm
import pl.metaprogramming.metamodel.oas.HttpResponse

import static pl.metaprogramming.codemodel.builder.java.ClassType.REST_RESPONSE_INTERFACE
import static pl.metaprogramming.codemodel.model.java.JavaDefs.*

class RestResponseStatusInterfaceBuildStrategy extends ClassCmBuildStrategy<Integer> {

    boolean withContent = true

    @Override
    void makeImplementation() {
        setInterface()
        addGenericParams(GENERIC_R)
        setSuperClass(getClass(REST_RESPONSE_INTERFACE).withGeneric(GENERIC_R))

        addMethods(makeIsResponseMethod())
        if (withContent) {
            addGenericParams(GENERIC_T)
            addMethods(makeGetResponseMethod())
        }
        if (!isStaticFactoryMethod()) {
            addAnnotation(ANNOT_PARAMETERS_ARE_NONNULL_BY_DEFAULT)
            addMethods(makeSetResponseMethod())
        }
    }

    private MethodCm makeGetResponseMethod() {
        new MethodCm(
                name: 'get' + modelName,
                modifiers: MODIFIER_DEFAULT,
                annotations: [ANNOT_SUPPRESS_UNCHECKED],
                resultType: GENERIC_T,
                implBody: codeBuf
                        .ifBlock("is${modelName}()",
                                "return ($GENERIC_T.className) getBody();")
                        .endBlock()
                        .addLines(makeGetResponseException())
                        .take()
        )
    }

    private String makeGetResponseException() {
        isDefaultResponse()
                ? "throw new IllegalStateException(String.format(\"Response other than %s is not set\", getDeclaredStatuses()));"
                : "throw new IllegalStateException(\"Response ${model} is not set\");"
    }

    private MethodCm makeIsResponseMethod() {
        new MethodCm(
                name: 'is' + modelName,
                modifiers: MODIFIER_DEFAULT,
                resultType: T_BOOLEAN_PRIMITIVE,
                implBody: (isDefaultResponse()
                        ? 'return !getDeclaredStatuses().contains(getStatus());'
                        : "return isStatus($model);") as String

        )
    }

    private MethodCm makeSetResponseMethod() {
        new MethodCm(
                name: 'set' + modelName,
                modifiers: MODIFIER_DEFAULT,
                resultType: GENERIC_R,
                params: makeSetResponseParams(),
                implBody: makeSetResponseImplBody()

        )
    }

    private List<FieldCm> makeSetResponseParams() {
        def params = []
        if (isDefaultResponse()) {
            params.add(new FieldCm(T_INTEGER, 'status'))
        }
        if (withContent) {
            params.add(new FieldCm(GENERIC_T, 'body'))
        }
        params
    }

    private String makeSetResponseImplBody() {
        if (isDefaultResponse()) {
            codeBuf.ifBlock("getDeclaredStatuses().contains(status)",
                'throw new IllegalArgumentException(String.format("Status %s is declared. Use dedicated setter for it.", status));'
            ).endBlock()
        }
        codeBuf.addLines("return set(${isDefaultResponse() ? 'status' : modelName}, ${withContent ? 'body' : 'null'});")
                .take()
    }

    private boolean isDefaultResponse() {
        model == HttpResponse.DEFAULT
    }

    @TypeChecked(TypeCheckingMode.SKIP)
    private boolean isStaticFactoryMethod() {
        getParams(SpringRestParams).staticFactoryMethodForRestResponse
    }

}
