/*
 * Copyright (c) 2019,2020 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.rest

import groovy.transform.TypeChecked
import groovy.transform.TypeCheckingMode
import pl.metaprogramming.codemodel.builder.java.ClassCmBuildStrategy
import pl.metaprogramming.codemodel.builder.java.config.SpringRestParams
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.MethodCm
import pl.metaprogramming.codemodel.model.java.ValueCm
import pl.metaprogramming.metamodel.oas.HttpResponse
import pl.metaprogramming.metamodel.oas.Operation

import static pl.metaprogramming.codemodel.builder.java.ClassType.*
import static pl.metaprogramming.codemodel.model.java.JavaDefs.*

class RestResponseBuildStrategy extends ClassCmBuildStrategy<Operation> {

    @Override
    void makeImplementation() {
        setSuperClass(getClass(REST_RESPONSE_ABSTRACT).withGeneric(classModel))
        if (isStaticFactoryMethod()) {
            addAnnotation(ANNOT_PARAMETERS_ARE_NONNULL_BY_DEFAULT)
            addConstructor()
        }
        addGetDeclaredStatuses()
        addMethods(new MethodCm(
                name: 'self',
                resultType: classModel,
                implBody: 'return this;'
        ))

        model.responses.each {
            def responseType = it.schema != null ? REST_RESPONSE_STATUS_INTERFACE : REST_RESPONSE_STATUS_NO_CONTENT_INTERFACE
            def genericParams = [classModel] as List<ClassCd>
            if (it.schema) {
                genericParams.add(getResponseClass(it))
            }
            addInterfaces(new ClassCd(getClass(responseType, it.status), genericParams))
            addFactoryMethod(it)
        }
    }

    private void addConstructor() {
        addMethods(new MethodCm(
                name: classModel.className,
                modifiers: MODIFIER_PRIVATE,
                resultType: classModel,
                params: [new FieldCm(T_INTEGER, 'status'),
                        new FieldCm(T_OBJECT, 'body')],
                implBody: 'super(status, body);'
        ))
    }

    private void addGetDeclaredStatuses() {
        addFields(new FieldCm(
                name: 'DECLARED_STATUSES',
                type: COLLECTION_OF_INT,
                modifiers: MODIFIER_PRIVATE_STATIC_FINAL,
                value: ValueCm.value(getDeclaredResponsesValue())
        ))
        addMethods(new MethodCm(
                name: 'getDeclaredStatuses',
                resultType: COLLECTION_OF_INT,
                implBody: 'return DECLARED_STATUSES;'
        ))
    }

    private String getDeclaredResponsesValue() {
        def statuses = model.responses.findAll {!it.default }.collect { it.status }
        if (statuses.empty) {
            addImports(COLLECTIONS)
            return 'Collections.emptyList()'
        }
        if (statuses.size() == 1) {
            addImports(COLLECTIONS)
            return "Collections.singletonList(${statuses[0]})"
        }
        addImports(ARRAYS)
        return "Arrays.asList(${statuses.join(', ')})"
    }

    private void addFactoryMethod(HttpResponse response) {
        if (!isStaticFactoryMethod()) {
            return
        }
        addMethods(new MethodCm(
                name: 'set' + (response.isDefault() ? 'Other' : response.status),
                modifiers: MODIFIER_PUBLIC_STATIC,
                resultType: classModel,
                params: getFactoryMethodParams(response),
                implBody: getFactoryMethodImpl(response)
        ))
    }

    private List<FieldCm> getFactoryMethodParams(HttpResponse response) {
        List<FieldCm> params = []
        if (response.isDefault()) {
            params.add(new FieldCm(T_INTEGER, 'status'))
        }
        if (response.schema != null) {
            params.add(new FieldCm(getResponseClass(response), 'body'))
        }
        params
    }

    private String getFactoryMethodImpl(HttpResponse response) {
        String status = response.isDefault() ? 'status' : '' + response.status
        String body = response.schema != null ? 'body' : 'null'
        if (response.isDefault()) {
            codeBuf.ifBlock("DECLARED_STATUSES.contains(status)",
                    'throw new IllegalArgumentException(String.format("Status %s is declared. Use dedicated factory method for it.", status));'
            ).endBlock()

        }
        codeBuf.addLines("return new ${classModel.className}($status, $body);")
                .take()
    }

    protected ClassCd getResponseClass(HttpResponse model) {
        model.schema == null ? null
                : getClass(model.schema.enumOrItemEnum ? ENUM : DTO, model.schema.dataType)
    }

    @TypeChecked(TypeCheckingMode.SKIP)
    private boolean isStaticFactoryMethod() {
        getParams(SpringRestParams).staticFactoryMethodForRestResponse
    }
}
