/*
 * Copyright (c) 2020 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.rest

import pl.metaprogramming.codemodel.builder.java.ClassCmBuildStrategy
import pl.metaprogramming.codemodel.builder.java.config.SpringRestParams
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.MethodCm

import static pl.metaprogramming.codemodel.builder.java.ClassType.REST_RESPONSE_INTERFACE
import static pl.metaprogramming.codemodel.model.java.JavaDefs.*

class RestResponseBaseBuildStrategy extends ClassCmBuildStrategy {

    @Override
    void makeImplementation() {
        addGenericParams(GENERIC_R)
        setModifiers(MODIFIER_PUBLIC_ABSTRACT)
        addInterfaces(getClass(REST_RESPONSE_INTERFACE).withGeneric(GENERIC_R))
        FieldCm[] fields = [
                new FieldCm(T_INTEGER, 'status'),
                new FieldCm(T_OBJECT, 'body'),
                new FieldCm(new ClassCd(T_MAP, [T_STRING, T_STRING]), 'headers').assign('new java.util.TreeMap<>()')
        ]
        fields.each {
            it.annotations = [ANNOT_LB_GETTER]
            it.modifiers = MODIFIER_PRIVATE_FINAL
        }
        addFields(fields)

        if (!getParams(SpringRestParams).staticFactoryMethodForRestResponse) {
            fields.findAll { ['status', 'body'].contains(it.name) }
                    .each { it.modifiers = MODIFIER_PRIVATE }
            addMethods(new MethodCm(
                    name: 'set',
                    annotations: [ANNOT_OVERRIDE],
                    params: [new FieldCm(name: 'status', type: T_INTEGER, annotations: [ANNOT_NON_NULL]),
                             new FieldCm(T_OBJECT, 'body')],
                    resultType: GENERIC_R,
                    implBody: codeBuf.ifBlock(
                            'this.status != null',
                            'throw new IllegalStateException(String.format("Response already initialized with %s - %s", this.status, this.body));')
                            .endBlock()
                            .addLines(
                                    "this.status = status;",
                                    "this.body = body;",
                                    "return self();").take()
            ))
        } else {
            addAnnotation(ANNOT_LB_REQUIRED_ARGS_CONTR)
        }
    }
}
