/*
 * Copyright (c) 2021 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.rest

import groovy.transform.TypeChecked
import groovy.transform.TypeCheckingMode
import pl.metaprogramming.codemodel.builder.java.ClassCmBuildStrategy
import pl.metaprogramming.codemodel.builder.java.config.ValidationParams
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.MethodCm

import static pl.metaprogramming.codemodel.builder.java.ClassType.*
import static pl.metaprogramming.codemodel.model.java.JavaDefs.GENERIC_V
import static pl.metaprogramming.codemodel.model.java.JavaDefs.MODIFIER_PUBLIC_ABSTRACT

class BaseValidatorBuildStrategy extends ClassCmBuildStrategy<Object> {

    @Override
    void makeImplementation() {
        setModifiers(MODIFIER_PUBLIC_ABSTRACT)
        addGenericParams(GENERIC_V)
        addInterfaces(getGenericClass(VALIDATION_CHECKER, [GENERIC_V]))
        addMethods new MethodCm(
                name: 'validate',
                resultType: getClass(VALIDATION_RESULT),
                params: [new FieldCm(GENERIC_V, 'input')],
                implBody: makeValidateImplBody()
        )
    }

    private String makeValidateImplBody() {
        codeBuf.addLines(
                'ValidationContext<V> context = new ValidationContext<>(input);',
                'check(context);',
        )
        if (throwExceptionIfValidationFailed) {
            def vex = getClass(VALIDATION_EXCEPTION)
            addImport(vex)
            codeBuf.ifBlock('!context.isValid()', "throw new ${vex.className}(context.getResult());").endBlock()
        }
        codeBuf.addLines('return context.getResult();')
        codeBuf.take()
    }

    @TypeChecked(TypeCheckingMode.SKIP)
    private boolean isThrowExceptionIfValidationFailed() {
        getParams(ValidationParams).throwExceptionIfValidationFailed
    }

}
