/*
 * Copyright (c) 2019,2020 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.module

import groovy.transform.TypeChecked
import groovy.transform.TypeCheckingMode
import pl.metaprogramming.codemodel.builder.java.ClassCmBuilder
import pl.metaprogramming.codemodel.builder.java.config.JavaModuleConfig
import pl.metaprogramming.codemodel.builder.java.config.JavaModuleParams
import pl.metaprogramming.codemodel.builder.java.config.SpringWsParams
import pl.metaprogramming.codemodel.builder.java.xml.XmlPackageInfoBuilder
import pl.metaprogramming.codemodel.formatter.JavaCodeFormatter
import pl.metaprogramming.metamodel.data.DataSchema
import pl.metaprogramming.metamodel.wsdl.WsdlApi

import static pl.metaprogramming.codemodel.builder.java.ClassType.*

class SpringWsOutAdapterBuilder extends JavaModuleBuilder<WsdlApi> {

    SpringWsOutAdapterBuilder(String moduleName, WsdlApi moduleModel, JavaModuleConfig config) {
        super(moduleName, moduleModel, config, CLASS_TYPE_BUILD_ORDER)
    }

    @Override
    void make() {
        addSchemaCodes(model)
        addPackageInfo(model)
        addObjectFactory(model.schemas.values())
        addOperationCodes(model)
        makeCodeModels()
    }

    void addPackageInfo(WsdlApi model) {
        def baseDir = getConfig(XML_DTO).baseDir
        assert baseDir == getConfig(XML_ENUM).baseDir
        def namespaces = model.schemas.values().collect { it.namespace }.unique()
        packageInfoList = namespaces.collect { namespace ->
            XmlPackageInfoBuilder.make(
                    baseDir,
                    namespace,
                    getPackage(namespace),
                    model.namespaceElementFormDefault[namespace],
                    config.params.get(JavaModuleParams).generatedAnnotation)
        }
    }

    void addObjectFactory(Collection<DataSchema> schemas) {
        schemas
                .findAll { !it.enum }
                .groupBy { it.namespace }
                .each { namespace, namespaceSchemas ->
                    addClass(namespace, XML_OBJECT_FACTORY, 'ObjectFactory', namespaceSchemas)
        }
    }

    void addOperationCodes(WsdlApi model) {
        String clientName = model.name.capitalize()
        addClass(WS_CLIENT, model, clientName)
        addClass(WS_CLIENT_CONFIGURATION, model, clientName)
    }

    void addSchemaCodes(WsdlApi model) {
        model.schemas.values().each { schema ->
            addClass(schema)
            model.schemaPackages.add(getPackage(schema.namespace))
        }
    }

    ClassCmBuilder addClass(DataSchema schema) {
        addClass(
                schema.namespace,
                schema.enum ? XML_ENUM : XML_DTO,
                JavaCodeFormatter.toJavaName(schema.code, true),
                schema.dataType
        )
    }

    ClassCmBuilder addClass(String namespace, def classType, String className, def model) {
        addClass(new ClassCmBuilder<>(
                model: model,
                modelName: className,
                packageName: getPackage(namespace)
        ), classType)
    }

    @TypeChecked(TypeCheckingMode.SKIP)
    String getPackage(String namespace) {
        def packageName = getParams(SpringWsParams).namespace2Package[namespace]
        assert packageName, "Package not defined for namespace: $namespace"
        packageName
    }

    private static final List CLASS_TYPE_BUILD_ORDER = [
            XML_ENUM,
            XML_DTO,
            WS_CLIENT,
            WS_CLIENT_CONFIGURATION,
            XML_OBJECT_FACTORY]
}
