/*
 * Copyright (c) 2020 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.module


import pl.metaprogramming.codemodel.builder.java.config.JavaModuleConfig
import pl.metaprogramming.codemodel.formatter.JavaCodeFormatter
import pl.metaprogramming.codemodel.model.java.JavaDefs
import pl.metaprogramming.metamodel.oas.Parameter
import pl.metaprogramming.metamodel.oas.RestApi

import static pl.metaprogramming.codemodel.builder.java.ClassType.*

class SpringRestOutAdapterBuilder extends JavaModuleBuilder<RestApi> {

    private static final List OPERATION_CLASSES = [REST_REQUEST_DTO, REQUEST_DTO, RESPONSE_DTO, REQUEST_MAPPER, RESPONSE_MAPPER]

    SpringRestOutAdapterBuilder(String moduleName, RestApi moduleModel, JavaModuleConfig config) {
        super(moduleName, moduleModel, config, CLASS_TYPE_BUILD_ORDER)
    }

    @Override
    void make() {
        addObjectParamsCodes(model.parameters)
        addSchemaCodes(model)
        addOperationCodes(model)
        makeCodeModels()
    }

    void addObjectParamsCodes(List<Parameter> parameters) {
        parameters.findAll { it.isObject() }.each { paramDef ->
            classIndex.put(JavaDefs.T_STRING, paramDef.dataType, REST_DTO)
            addClass(DTO, paramDef.dataType, paramDef.objectType.code)
            addClass(REST_MAPPER, paramDef.dataType, paramDef.objectType.code)
        }
    }

    void addSchemaCodes(RestApi model) {
        model.schemas.each { schema ->
            if (schema.isEnum()) {
                addClass(ENUM, schema)
            }
            if (schema.isObject()) {
                addClass(DTO, schema)
                addClass(REST_DTO, schema)
                addClass(REST_MAPPER, schema)
            }
        }
    }

    void addOperationCodes(RestApi model) {
        model.groupedOperations.each { group, operations ->
            String groupJavaName = JavaCodeFormatter.toCapitalizeJavaName(group)
            operations.each { operation ->
                String dataName = operation.code.capitalize()
                OPERATION_CLASSES.each { classType ->
                    addClass(classType, operation, dataName)
                }
            }
            addClass(REST_CLIENT, operations, groupJavaName)
            addClass(REST_CLIENT_IMPL, operations, groupJavaName)
        }
    }

    private static final List CLASS_TYPE_BUILD_ORDER = [
            ENUM,
            DTO,
            REST_DTO,
            REQUEST_DTO,
            REST_REQUEST_DTO,
            RESPONSE_DTO,
            REST_MAPPER,
            REQUEST_MAPPER,
            RESPONSE_MAPPER,
            REST_CLIENT_IMPL,
            REST_CLIENT]
}
