/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.module

import groovy.transform.TypeChecked
import groovy.transform.TypeCheckingMode
import pl.metaprogramming.codemodel.builder.java.config.JavaModuleConfig
import pl.metaprogramming.codemodel.builder.java.config.SpringRestParams
import pl.metaprogramming.codemodel.formatter.JavaCodeFormatter
import pl.metaprogramming.codemodel.model.java.JavaDefs
import pl.metaprogramming.metamodel.oas.Operation
import pl.metaprogramming.metamodel.oas.Parameter
import pl.metaprogramming.metamodel.oas.RestApi

import java.util.function.Consumer

import static pl.metaprogramming.codemodel.builder.java.ClassType.*

class SpringRestInAdapterBuilder extends JavaModuleBuilder<RestApi> {

    private static final List OPERATION_DTO_CLASSES = [REST_REQUEST_DTO, REQUEST_DTO, RESPONSE_DTO]

    private Consumer<RestOperationBuildHelper> additionalOperationCodeBuilder

    SpringRestInAdapterBuilder(String moduleName, RestApi moduleModel, JavaModuleConfig config, Consumer<RestOperationBuildHelper> additionalOperationCodeBuilder = {}) {
        super(moduleName, moduleModel, config, CLASS_TYPE_BUILD_ORDER)
        this.additionalOperationCodeBuilder = additionalOperationCodeBuilder
    }

    @Override
    void make() {
        addObjectParamsCodes(model.parameters)
        addSchemaCodes(model)
        addOperationCodes(model)
        makeCodeModels()
    }

    void addObjectParamsCodes(List<Parameter> parameters) {
        parameters.findAll { it.isObject() }.each { paramDef ->
            classIndex.put(JavaDefs.T_STRING, paramDef.dataType, REST_DTO)
            addClass(DTO, paramDef.dataType, paramDef.objectType.code)
            addClass(REST_MAPPER, paramDef.dataType, paramDef.objectType.code)
        }
    }

    void addSchemaCodes(RestApi model) {
        model.schemas.each { schema ->
            if (schema.isEnum()) {
                addClass(ENUM, schema)
            }
            if (schema.isObject()) {
                addClass(DTO, schema)
                addClass(REST_DTO, schema)
                addClass(REST_MAPPER, schema)
                addClass(REST_DTO_VALIDATOR, schema)
            }
        }
    }

    void addOperationCodes(RestApi model) {
        model.groupedOperations.each { group, operations ->
            String javaGroupName = JavaCodeFormatter.toCapitalizeJavaName(group)
            operations.each { operation ->
                String dataName = operation.code.capitalize()
                operationClassTypes().each { classType ->
                    addClass(classType, operation, dataName)
                }
                buildAdditionalOperationCodes(operation)
            }
            resourceClassTypes().each { classType ->
                addClass(classType, operations, javaGroupName)
            }
        }
    }

    void buildAdditionalOperationCodes(Operation operation) {
        additionalOperationCodeBuilder.accept(new RestOperationBuildHelper(generationTasksBuilder: this, operation: operation))
    }


    @TypeChecked(TypeCheckingMode.SKIP)
    private boolean isMultiOperationControllerStrategy() {
        !getParams(SpringRestParams).controllerPerOperation
    }

    private List operationClassTypes() {
        def result = OPERATION_DTO_CLASSES + [REQUEST_MAPPER, REST_REQUEST_VALIDATOR, RESPONSE_MAPPER]
        if (!isMultiOperationControllerStrategy()) {
            result.add(REST_CONTROLLER)
        }
        result
    }

    private List resourceClassTypes() {
        def result = [FACADE, FACADE_IMPL]
        if (isMultiOperationControllerStrategy()) {
            result.add(REST_CONTROLLER_MO)
        }
        result
    }

    static class RestOperationBuildHelper {
        private JavaModuleBuilder<RestApi> generationTasksBuilder
        private Operation operation

        void addClass(def classType, String modelName = defaultModelName()) {
            generationTasksBuilder.addClass(classType, operation, modelName).make()
        }

        String defaultModelName() {
            operation.group.capitalize() + operation.code.capitalize()
        }
    }

    private static final List CLASS_TYPE_BUILD_ORDER = [
            ENUM,
            DTO,
            REST_DTO,
            REQUEST_DTO,
            REST_REQUEST_DTO,
            RESPONSE_DTO,
            FACADE_IMPL,
            FACADE,
            REST_MAPPER,
            REQUEST_MAPPER,
            RESPONSE_MAPPER,
            REST_DTO_VALIDATOR,
            REST_REQUEST_VALIDATOR,
            REST_CONTROLLER,
            REST_CONTROLLER_MO]
}
