/*
 * Copyright (c) 2019,2020 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.module

import pl.metaprogramming.codemodel.builder.CodeGenerationTask
import pl.metaprogramming.codemodel.builder.ModuleBuilder
import pl.metaprogramming.codemodel.builder.java.ClassBuilderConfigurator
import pl.metaprogramming.codemodel.builder.java.ClassCmBuilder
import pl.metaprogramming.codemodel.builder.java.config.JavaModuleConfig
import pl.metaprogramming.codemodel.builder.java.config.JavaModuleParams
import pl.metaprogramming.codemodel.formatter.PackageInfoFormatter
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.PackageInfoCm
import pl.metaprogramming.codemodel.model.java.index.ClassEntry
import pl.metaprogramming.codemodel.model.java.index.ClassIndex
import pl.metaprogramming.metamodel.data.DataSchema
import pl.metaprogramming.metamodel.data.NamedDataType

class JavaModuleBuilder<T> implements ModuleBuilder<T> {

    T model
    String moduleName
    JavaModuleConfig config
    ClassIndex classIndex
    List<PackageInfoCm> packageInfoList = []

    JavaModuleBuilder(String moduleName, T moduleModel, JavaModuleConfig config) {
        this(moduleName, moduleModel, config, new ArrayList(config.classBuilderConfigs.keySet()))
    }

    JavaModuleBuilder(String moduleName, T moduleModel, JavaModuleConfig config, List classTypeBuildOrder) {
        this.moduleName = moduleName
        this.config = config
        this.model = moduleModel
        classIndex = new ClassIndex(moduleName, classTypeBuildOrder)
    }

    ModuleBuilder dependsOn(ClassIndex... dependsOn) {
        classIndex.dependsOn = Arrays.asList(dependsOn)
        this
    }

    @Override
    void make() {
        config.classBuilderConfigs.values().each {
            try {
                it.register(classIndex, config.params)
            } catch (Exception e) {
                throw new IllegalStateException("Can't register class: " + it.classType, e)
            }
        }
        makeCodeModels()
    }

    @Override
    List<CodeGenerationTask> getCodesToGenerate() {
        if (config.params.skipGeneration) {
            return []
        }
        makeClassGenerationTasksFromClassIndex() + makePackageInfoGenerationsTasks()
    }

    protected ModuleBuilder makeCodeModels() {
        classIndex.makeCodeModels()
        this
    }

    protected List<CodeGenerationTask> makeClassGenerationTasksFromClassIndex() {
        classIndex.codesToGenerate.collect {
            it.builder.makeDecoration()
            makeCodeGenerationTask(it)
        }
    }

    protected CodeGenerationTask makeCodeGenerationTask(ClassEntry entry) {
        assert entry.clazz.packageName
        try {
            def result = entry.builder.makeGenerationTask(getClassFilePath(entry.classType, entry.clazz))
            result.codeDecorators = getParams(JavaModuleParams).codeDecorators
            result
        } catch (Exception e) {
            throw new IllegalStateException("Can't make CodeGenerationTask for: ${entry.classType}", e)
        }
    }

    protected List<CodeGenerationTask> makePackageInfoGenerationsTasks() {
        packageInfoList.collect {
            new CodeGenerationTask(
                    destFilePath: getJavaFilePath(it.baseDir, it.packageName, 'package-info'),
                    codeModel: it,
                    formatter: new PackageInfoFormatter()
            )
        }
    }

    protected String getClassFilePath(Object classType, ClassCd classCd) {
        def baseDir = getConfig(classType).baseDir
        assert baseDir, "No base dir for class type: $classType"
        getJavaFilePath(baseDir, classCd.packageName, classCd.className)
    }

    static protected String getJavaFilePath(String baseDir, String packageName, String className) {
        "$baseDir/${pkgToDir(packageName)}/${className}.java"
    }

    protected ClassCmBuilder addClass(def classType, def model, String modelName) {
        getConfig(classType).register(classIndex, config.params, modelName, model) as ClassCmBuilder
    }

    protected ClassCmBuilder addClass(Object classType, DataSchema schema) {
        def dataType = schema.dataType as NamedDataType
        addClass(classType, dataType, dataType.code)
    }

    protected ClassCmBuilder addClass(ClassCmBuilder builder, Object classType) {
        builder.config = getConfig(classType)
        builder.params = config.params
        builder.classIndex = classIndex
        classIndex.put(builder)
        builder
    }

    protected ClassBuilderConfigurator getConfig(def classType) {
        config.getConfig(classType)
    }

    protected <T> T getParams(Class<T> clazz) {
        config.params.get(clazz)
    }

    static String pkgToDir(String pkg) {
        pkg.replaceAll('\\.', '/')
    }
}