/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.mapper

import pl.metaprogramming.codemodel.builder.java.ClassCmBuildStrategy
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.MethodCm

import static pl.metaprogramming.codemodel.model.java.JavaDefs.*

class SerializationUtilsBuildStrategy extends ClassCmBuildStrategy {

    static final ClassCd T_COLLECTORS = new ClassCd('java.util.stream.Collectors')
    static final ClassCd T_ZONED_DATE_TIME = new ClassCd('java.time.ZonedDateTime')
    static final ClassCd T_ZONED_OFFSET = new ClassCd('java.time.ZoneOffset')
    static final ClassCd T_DATA_TIME_FORMATTER = new ClassCd('java.time.format.DateTimeFormatter')

    List<MethodCm> makePublicStatic(List<MethodCm> methods) {
        methods.each {
            it.modifiers = MODIFIER_PUBLIC_STATIC
        }
    }

    @Override
    void makeDeclaration() {
        addMethods(makePublicStatic(makeGenericMethods()))
        addMappers(makePublicStatic(makeMethods()))
    }

    List<MethodCm> makeGenericMethods() {
        [new MethodCm(name: 'fromString', resultType: GENERIC_T,
                params: [new FieldCm(T_STRING, 'value'), new FieldCm(new ClassCd(FUN, [T_STRING, GENERIC_T]), 'transformer')],
                implBody: 'return value == null || value.isEmpty() ? null : transformer.apply(value);'
        ),
         new MethodCm(name: 'toString', resultType: T_STRING,
                 params: [new FieldCm(GENERIC_T, 'value'), new FieldCm(new ClassCd(FUN, [GENERIC_T, T_STRING]), 'transformer')],
                 implBody: 'return value == null ? null : transformer.apply(value);'
         )]
    }

    List<MethodCm> makeMethods() {
        def rawParam = [new FieldCm(T_STRING, 'value')]
        def formatParam = new FieldCm(T_STRING, 'format')
        [new MethodCm(name: 'transformList', resultType: LIST_R,
                params: [new FieldCm(LIST_T, 'value'), new FieldCm(FUN_T_R, 'transformer')],
                implBody: 'return value == null ? null : value.stream().map(transformer).collect(Collectors.toList());',
                implDependencies: [T_COLLECTORS]
        ),
         new MethodCm(name: 'transformMap', resultType: MAP_KR,
                 params: [new FieldCm(MAP_KT, 'value'), new FieldCm(FUN_T_R, 'transformer')],
                 implBody: 'return value == null ? null : value.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> transformer.apply(e.getValue())));',
                 implDependencies: [T_COLLECTORS]
         ),
         new MethodCm(name: 'toLong', resultType: T_LONG, params: rawParam,
                 implBody: 'return fromString(value, v -> Long.valueOf(v));'
         ),
         new MethodCm(name: 'toString', resultType: T_STRING, params: [new FieldCm(T_LONG, 'value')],
                 implBody: 'return toString(value, v -> v.toString());'
         ),
         new MethodCm(name: 'toInteger', resultType: T_INTEGER, params: rawParam,
                 implBody: 'return fromString(value, v -> Integer.valueOf(v));'
         ),
         new MethodCm(name: 'toString', resultType: T_STRING, params: [new FieldCm(T_INTEGER, 'value')],
                 implBody: 'return toString(value, v -> v.toString());'
         ),
         new MethodCm(name: 'toFloat', resultType: T_FLOAT, params: rawParam,
                 implBody: 'return fromString(value, v -> Float.valueOf(v));'
         ),
         new MethodCm(name: 'toString', resultType: T_STRING, params: [new FieldCm(T_FLOAT, 'value')],
                 implBody: 'return toString(value, v -> v.toString());'
         ),
         new MethodCm(name: 'toBigDecimal', resultType: T_BIG_DECIMAL, params: rawParam,
                 implBody: 'return fromString(value, v -> new BigDecimal(v));',
                 implDependencies: [T_BIG_DECIMAL]
         ),
         new MethodCm(name: 'toString', resultType: T_STRING, params: [new FieldCm(T_BIG_DECIMAL, 'value')],
                 implBody: 'return toString(value, v -> v.toString());',
                 implDependencies: [T_BIG_DECIMAL]
         ),
         new MethodCm(name: 'toBigDecimal', resultType: T_BIG_DECIMAL, params: rawParam + formatParam,
                 implBody: 'return fromString(value, v -> new BigDecimal(v));',
                 implDependencies: [T_BIG_DECIMAL]
         ),
         new MethodCm(name: 'toString', resultType: T_STRING, params: [new FieldCm(T_BIG_DECIMAL, 'value')] + formatParam,
                 implBody: 'return toString(value, v -> v.toString());',
                 implDependencies: [T_BIG_DECIMAL]
         ),

         new MethodCm(name: 'toLocalDate', resultType: T_LOCAL_DATE, params: [new FieldCm(T_STRING, 'value')],
                 implBody: 'return fromString(value, v -> LocalDate.parse(v));',
                 implDependencies: [T_LOCAL_DATE]
         ),

         new MethodCm(name: 'toString', resultType: T_STRING, params: [new FieldCm(T_LOCAL_DATE, 'value')],
                 implBody: 'return toString(value, v -> DateTimeFormatter.ISO_LOCAL_DATE.format(v));',
                 implDependencies: [T_DATA_TIME_FORMATTER]
         ),

         new MethodCm(name: 'toLocalDateTime', resultType: T_LOCAL_DATE_TIME, params: [new FieldCm(T_STRING, 'value')],
                 implBody: 'return fromString(value, v -> ZonedDateTime.parse(v, DateTimeFormatter.ISO_OFFSET_DATE_TIME).withZoneSameInstant(ZoneOffset.UTC).toLocalDateTime());',
                 implDependencies: [T_ZONED_DATE_TIME, T_ZONED_OFFSET, T_DATA_TIME_FORMATTER]
         ),

         new MethodCm(name: 'toString', resultType: T_STRING, params: [new FieldCm(T_LOCAL_DATE_TIME, 'value')],
                 implBody: 'return toString(value, v -> DateTimeFormatter.ISO_OFFSET_DATE_TIME.format(value.atZone(ZoneOffset.UTC)));',
                 implDependencies: [T_ZONED_OFFSET, T_DATA_TIME_FORMATTER]

         ),

         new MethodCm(name: 'toString', resultType: T_STRING, params: [new FieldCm(T_ZONED_DATE_TIME, 'value')],
                 implBody: 'return toString(value, v -> DateTimeFormatter.ISO_OFFSET_DATE_TIME.format(value));',
                 implDependencies: [T_DATA_TIME_FORMATTER]

         ),

         new MethodCm(name: 'toBoolean', resultType: T_BOOLEAN, params: rawParam,
                 implBody: 'return fromString(value, v -> Boolean.valueOf(v));'
         ),
         new MethodCm(name: 'toString', resultType: T_STRING, params: [new FieldCm(T_BOOLEAN, 'value')],
                 implBody: 'return toString(value, v -> v.toString());'
         ),

         new MethodCm(name: 'toDouble', resultType: T_DOUBLE, params: rawParam,
                 implBody: 'return fromString(value, v -> Double.valueOf(v));'
         ),
         new MethodCm(name: 'toString', resultType: T_STRING, params: [new FieldCm(T_DOUBLE, 'value')],
                 implBody: 'return toString(value, v -> v.toString());'
         ),
        ]
    }

}
