/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.dto


import pl.metaprogramming.codemodel.model.java.AnnotationCm
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.ValueCm
import pl.metaprogramming.metamodel.data.DataSchema
import pl.metaprogramming.metamodel.data.DataType

import static pl.metaprogramming.codemodel.builder.java.ClassType.MAP_RAW_VALUE_SERIALIZER
import static pl.metaprogramming.codemodel.builder.java.ClassType.REST_DTO
import static pl.metaprogramming.codemodel.builder.java.MetaModelAttribute.JAVA_NAME

class JsonDtoBuildStrategy extends BaseDtoBuildStrategy {

    static final String CLASS_ANNOTATION = 'com.fasterxml.jackson.annotation.JsonAutoDetect'
    static final String FIELD_ANNOTATION = 'com.fasterxml.jackson.annotation.JsonProperty'
    static final AnnotationCm ANNOTATION = new AnnotationCm(CLASS_ANNOTATION, [fieldVisibility: ValueCm.value('JsonAutoDetect.Visibility.NONE')])
    static final AnnotationCm JSON_RAW_VALUE_ANNOTATION = new AnnotationCm('com.fasterxml.jackson.annotation.JsonRawValue')

    private boolean withDescriptiveFields = true

    @Override
    void makeImplementation() {
        super.makeImplementation()
        addAnnotation(ANNOTATION)
    }

    void addField(DataSchema schema) {
        def fieldCm = new FieldCm(
                model: schema,
                name: schema.getAdditive(JAVA_NAME),
                type: getClass(REST_DTO, schema.dataType),
                annotations: createFieldAnnotations(schema))
        if (schema.defaultValue) {
            fieldCm.assign(ValueCm.escaped(schema.defaultValue))
        }
        if (withDescriptiveFields) {
            addDescriptiveFields(schema, fieldCm)
        }
        addFields(fieldCm)
    }

    List<AnnotationCm> createFieldAnnotations(DataSchema schema) {
        def result = [new AnnotationCm(FIELD_ANNOTATION, [value: ValueCm.escaped(schema.code)])]
        if (isNumberOrBoolean(schema.baseDataType)) {
            result.add(JSON_RAW_VALUE_ANNOTATION)
        }
        if (schema.isMap() && isNumberOrBoolean(schema.mapType.valuesSchema.dataType)) {
            def serializer = getClass(MAP_RAW_VALUE_SERIALIZER)
            addImports(serializer)
            result.add(new AnnotationCm('com.fasterxml.jackson.databind.annotation.JsonSerialize', ['using': ValueCm.value(serializer.className + '.class')]))
        }
        result
    }

    private static boolean isNumberOrBoolean(DataType dataType) {
        DataType.BOOLEAN == dataType || DataType.NUMBER_TYPES.contains(dataType)
    }
}
