/*
 * Copyright (c) 2019,2020 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.dto

import pl.metaprogramming.codemodel.builder.java.ClassCmBuildStrategy
import pl.metaprogramming.codemodel.formatter.JavaCodeFormatter
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.ValueCm
import pl.metaprogramming.metamodel.data.DataSchema
import pl.metaprogramming.metamodel.data.ObjectType
import pl.metaprogramming.metamodel.oas.Parameter

import static pl.metaprogramming.codemodel.builder.java.ClassType.VALIDATION_FIELD
import static pl.metaprogramming.codemodel.builder.java.MetaModelAttribute.DESCRIPTION_FIELD_NAME
import static pl.metaprogramming.codemodel.builder.java.MetaModelAttribute.JAVA_NAME

abstract class BaseDtoBuildStrategy extends ClassCmBuildStrategy<ObjectType> {

    @Override
    void makeImplementation() {
        if (model.inherits) {
            assert model.inherits.size() == 1
            setSuperClass(classType, model.inherits[0])
        }
        addFields()
    }

    void addFields() {
        model.fields.each { field ->
            try {
                field.setAdditiveIfNotSet(JAVA_NAME, { JavaCodeFormatter.toJavaName(field.code) })
                addField(field)
            } catch (Exception e) {
                throw new IllegalStateException("Can't handle field '$field.code' for model '$modelName'", e)
            }
        }
    }

    abstract void addField(DataSchema schema)

    protected void addDescriptiveFields(DataSchema schema, FieldCm fieldCm) {
        def originalName = schema instanceof Parameter ? "$schema.name ($schema.location parameter)" : schema.code
        def descriptionFieldName = 'FIELD_' + JavaCodeFormatter.toUpperCase(fieldCm.name)
        schema.setAdditiveIfNotSet(DESCRIPTION_FIELD_NAME, { descriptionFieldName })
        def fieldClass = getClass(VALIDATION_FIELD)
        String value = "new ${fieldClass.className}<>(" +
                "\"$originalName\", " +
                "${classModel.className}::get${fieldCm.name.capitalize()})"

        addFields new FieldCm(
                name: descriptionFieldName,
                modifiers: 'public static final',
                type: new ClassCd(fieldClass, [classModel, fieldCm.type]),
                value: ValueCm.value(value)
        )
    }
}
