/*
 * Copyright (c) 2020 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.config

import pl.metaprogramming.codemodel.builder.java.base.EnumBuildStrategy
import pl.metaprogramming.codemodel.builder.java.base.InterfaceBuildStrategy
import pl.metaprogramming.codemodel.builder.java.dto.DtoBuildStrategy
import pl.metaprogramming.codemodel.builder.java.dto.JsonDtoBuildStrategy
import pl.metaprogramming.codemodel.builder.java.dto.RawDtoBuildStrategy
import pl.metaprogramming.codemodel.builder.java.module.SpringRestOutAdapterBuilder
import pl.metaprogramming.codemodel.builder.java.rest.RestDtoMapperBuildStrategy
import pl.metaprogramming.codemodel.builder.java.rest.RestResponseBuildStrategy
import pl.metaprogramming.codemodel.builder.java.spring.PayloadFieldAppenderBuildStrategy
import pl.metaprogramming.codemodel.builder.java.spring.RestClientImplBuildStrategy
import pl.metaprogramming.codemodel.builder.java.spring.SpringDefs
import pl.metaprogramming.codemodel.builder.java.spring.mapper.RestOutRequestMapperBuildStrategy
import pl.metaprogramming.codemodel.builder.java.spring.mapper.RestOutResponseMapperBuildStrategy
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.JavaDefs
import pl.metaprogramming.codemodel.model.java.index.DataTypeMapper
import pl.metaprogramming.metamodel.data.DataType
import pl.metaprogramming.metamodel.data.ObjectType
import pl.metaprogramming.metamodel.oas.Operation
import pl.metaprogramming.metamodel.oas.RestApi

import java.util.function.Function

import static pl.metaprogramming.codemodel.builder.java.ClassType.*

class SpringRestOutAdapterConfigurator extends JavaModuleConfigurator<SpringRestOutAdapterConfigurator> {

    static final Function<Operation, ObjectType> OPERATION_REQUEST_SCHEMA = { Operation o -> o.requestSchema } as Function<Operation, ObjectType>

    static final DataTypeMapper REST_DATA_TYPE_MAPPER = new DataTypeMapper({ DataType it ->
        it.isBinary() ? SpringDefs.RESOURCE : !it.isComplex() ? JavaDefs.T_STRING : null
    } as Function<DataType, ClassCd>)

    SpringRestOutAdapterConfigurator(CodegenParams params = new CodegenParams()) {
        super(params.withIfNotSet(SpringRestParams.DEFAULTS))
        dataTypeMapper.setMapping(DataType.BINARY, SpringDefs.RESOURCE)
        init()
    }

    SpringRestOutAdapterBuilder makeBuilder(String moduleName, RestApi moduleModel) {
        new SpringRestOutAdapterBuilder(moduleName, moduleModel, moduleConfig)
    }

    private void init() {
        generateIfUsed = true
        addClass(ENUM, 'Enum', new EnumBuildStrategy())
        addClass(RESPONSE_DTO, 'Response', new RestResponseBuildStrategy())
        addComponent(RESPONSE_MAPPER, 'ResponseMapper', new RestOutResponseMapperBuildStrategy())

        addLombokData(DTO, 'Dto', new DtoBuildStrategy())
        addLombokData(REST_DTO, 'Rdto', new JsonDtoBuildStrategy(withDescriptiveFields: false))
        addComponent(REST_MAPPER, 'Mapper', new RestDtoMapperBuildStrategy())

        addLombokData(REQUEST_DTO, 'Request', new DtoBuildStrategy(modelMapper: OPERATION_REQUEST_SCHEMA))
        addLombokData(REST_REQUEST_DTO, 'Rrequest', new PayloadFieldAppenderBuildStrategy(), new RawDtoBuildStrategy(withDescriptiveFields: false, modelMapper: OPERATION_REQUEST_SCHEMA))
        addComponent(REQUEST_MAPPER, 'RequestMapper', new RestOutRequestMapperBuildStrategy())

        addClass(REST_CLIENT, 'Client', new InterfaceBuildStrategy(REST_CLIENT_IMPL))
        addComponent(REST_CLIENT_IMPL, 'ClientImpl', new RestClientImplBuildStrategy())

        dataTypeMapper.setMapper(REST_DATA_TYPE_MAPPER, REST_DTO, REST_REQUEST_DTO)
        generateAlways(REST_CLIENT, REST_CLIENT_IMPL)
    }

    SpringRestOutAdapterConfigurator setRootPackage(String rootPackage) {
        setPackages(rootPackage, 'rest')
    }

    SpringRestOutAdapterConfigurator setPackages(String rootPackage, String adapterName) {
        super.setRootPackage(rootPackage)
        setPackage("${rootPackage}.ports.out.${adapterName}.dtos", DTO, ENUM, REQUEST_DTO, RESPONSE_DTO)
        setPackage("${rootPackage}.ports.out.${adapterName}", REST_CLIENT)
        setPackage("${rootPackage}.adapters.out.${adapterName}.dtos", REST_DTO, REST_REQUEST_DTO)
        setPackage("${rootPackage}.adapters.out.${adapterName}.mappers", REST_MAPPER, REQUEST_MAPPER, RESPONSE_MAPPER)
        setPackage("${rootPackage}.adapters.out.${adapterName}", REST_CLIENT_IMPL)
        this
    }

}
