/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.config

import pl.metaprogramming.codemodel.builder.java.base.EnumBuildStrategy
import pl.metaprogramming.codemodel.builder.java.base.InterfaceBuildStrategy
import pl.metaprogramming.codemodel.builder.java.dto.DtoBuildStrategy
import pl.metaprogramming.codemodel.builder.java.dto.JsonDtoBuildStrategy
import pl.metaprogramming.codemodel.builder.java.dto.RawDtoBuildStrategy
import pl.metaprogramming.codemodel.builder.java.module.SpringRestInAdapterBuilder
import pl.metaprogramming.codemodel.builder.java.rest.*
import pl.metaprogramming.codemodel.builder.java.spring.PayloadFieldAppenderBuildStrategy
import pl.metaprogramming.codemodel.builder.java.spring.RestControllerBuildStrategy
import pl.metaprogramming.codemodel.builder.java.spring.RestControllerMoBuildStrategy
import pl.metaprogramming.codemodel.builder.java.spring.SpringDefs
import pl.metaprogramming.codemodel.builder.java.spring.mapper.RestInRequestMapperBuildStrategy
import pl.metaprogramming.codemodel.builder.java.spring.mapper.RestInResponseMapperBuildStrategy
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.JavaDefs
import pl.metaprogramming.codemodel.model.java.index.DataTypeMapper
import pl.metaprogramming.metamodel.data.DataType
import pl.metaprogramming.metamodel.data.ObjectType
import pl.metaprogramming.metamodel.oas.Operation
import pl.metaprogramming.metamodel.oas.RestApi

import java.util.function.Consumer
import java.util.function.Function

import static pl.metaprogramming.codemodel.builder.java.ClassType.*

class SpringRestInAdapterConfigurator extends JavaModuleConfigurator<SpringRestInAdapterConfigurator> {

    static final Function<Operation, ObjectType> OPERATION_REQUEST_SCHEMA = { Operation o -> o.requestSchema } as Function<Operation, ObjectType>

    static final DataTypeMapper REST_DATA_MAPPING = new DataTypeMapper({DataType it ->
        it.isBinary() ? SpringDefs.MULTIPART_FILE : !it.isComplex() ? JavaDefs.T_STRING : null
    } as Function<DataType, ClassCd>)

    SpringRestInAdapterConfigurator(CodegenParams params = new CodegenParams()) {
        super(params.withIfNotSet(SpringRestParams.DEFAULTS))
        dataTypeMapper.setMapper(REST_DATA_MAPPING, REST_DTO, REST_REQUEST_DTO, REST_CONTROLLER_MO)
        params.get(ValidationParams).formatValidators
                .setIfNotPresent('date', 'field:VALIDATION_COMMON_CHECKERS.ISO_DATE')
                .setIfNotPresent('date-time', 'field:VALIDATION_COMMON_CHECKERS.ISO_DATE_TIME')
        init()
    }

    SpringRestInAdapterBuilder makeBuilder(String moduleName, RestApi moduleModel, Consumer<SpringRestInAdapterBuilder.RestOperationBuildHelper> additionalOperationCodeBuilder = {}) {
        new SpringRestInAdapterBuilder(moduleName, moduleModel, moduleConfig, additionalOperationCodeBuilder)
    }

    private void init() {
        generateIfUsed = true
        addComponent(REST_REQUEST_VALIDATOR, 'Validator', new RestRequestValidatorBuildStrategy(
                dtoTypeOfCode: REST_REQUEST_DTO,
                validatorTypeOfCode: REST_DTO_VALIDATOR,
                enumTypeOfCode: ENUM,
                addDataTypeValidations: true
        ))
        addComponent(REST_DTO_VALIDATOR, 'Validator', new RestDtoValidatorBuildStrategy(
                dtoTypeOfCode: REST_DTO,
                validatorTypeOfCode: REST_DTO_VALIDATOR,
                enumTypeOfCode: ENUM,
                addDataTypeValidations: true
        ))

        addClass(ENUM, 'Enum', new EnumBuildStrategy())
        addClass(RESPONSE_DTO, 'Response', new RestResponseBuildStrategy())
        addComponent(RESPONSE_MAPPER, 'ResponseMapper', new RestInResponseMapperBuildStrategy())

        addLombokData(DTO, 'Dto', new DtoBuildStrategy())
        addLombokData(REST_DTO, 'Rdto', new JsonDtoBuildStrategy())
        addComponent(REST_MAPPER, 'Mapper', new RestDtoMapperBuildStrategy())

        addLombokData(REQUEST_DTO, 'Request', new DtoBuildStrategy(modelMapper: OPERATION_REQUEST_SCHEMA))
        addLombokData(REST_REQUEST_DTO, 'Rrequest', new PayloadFieldAppenderBuildStrategy(), new RawDtoBuildStrategy(modelMapper: OPERATION_REQUEST_SCHEMA))
        addComponent(REQUEST_MAPPER, 'RequestMapper', new RestInRequestMapperBuildStrategy())

        addClass(REST_CONTROLLER, 'Controller', new RestControllerBuildStrategy(), diStrategy)
        addClass(REST_CONTROLLER_MO, 'Controller', new RestControllerMoBuildStrategy(), diStrategy)
        addClass(FACADE, 'Facade', new InterfaceBuildStrategy(FACADE_IMPL))
        addComponent(FACADE_IMPL, 'FacadeImpl', new FacadeBuildStrategy())

        generateAlways(FACADE, FACADE_IMPL, REST_CONTROLLER, REST_CONTROLLER_MO)
    }

    SpringRestInAdapterConfigurator setRootPackage(String rootPackage) {
        super.setRootPackage(rootPackage)
        setPackage(rootPackage + '.application', FACADE_IMPL)
        setPackage(rootPackage + '.ports.in.rest.dtos', DTO, ENUM, REQUEST_DTO, RESPONSE_DTO)
        setPackage(rootPackage + '.ports.in.rest', FACADE)
        setPackage(rootPackage + '.adapters.in.rest.dtos', REST_DTO, REST_REQUEST_DTO)
        setPackage(rootPackage + '.adapters.in.rest.mappers', REST_MAPPER, REQUEST_MAPPER)
        setPackage(rootPackage + '.adapters.in.rest.validators', REST_DTO_VALIDATOR)
        if (params.get(SpringRestParams).controllerPerOperation) {
            setPackage(rootPackage + '.adapters.in.rest.${group}', REST_REQUEST_VALIDATOR, REST_CONTROLLER, RESPONSE_MAPPER)
        } else {
            setPackage(rootPackage + '.adapters.in.rest.validators', REST_REQUEST_VALIDATOR)
            setPackage(rootPackage + '.adapters.in.rest.mappers', RESPONSE_MAPPER)
            setPackage(rootPackage + '.adapters.in.rest', REST_CONTROLLER_MO)
        }
        this
    }

}
