/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.config

import pl.metaprogramming.codegen.validation.DictionaryCheckerBuildStrategy
import pl.metaprogramming.codemodel.builder.java.ClassBuilderConfigurator
import pl.metaprogramming.codemodel.builder.java.base.BaseEnumBuildStrategy
import pl.metaprogramming.codemodel.builder.java.base.EnumValueInterfaceBuilder
import pl.metaprogramming.codemodel.builder.java.base.PlainEnumBuildStrategy
import pl.metaprogramming.codemodel.builder.java.base.PrivateConstructorBuildStrategy
import pl.metaprogramming.codemodel.builder.java.mapper.SerializationUtilsBuildStrategy
import pl.metaprogramming.codemodel.builder.java.module.JavaModuleBuilder
import pl.metaprogramming.codemodel.builder.java.rest.*
import pl.metaprogramming.codemodel.builder.java.spring.EndpointProviderBuildStrategy
import pl.metaprogramming.codemodel.builder.java.spring.SpringDefs
import pl.metaprogramming.codemodel.builder.java.spring.mapper.ValidationResultMapperBuildStrategy
import pl.metaprogramming.metamodel.oas.HttpResponse

import static pl.metaprogramming.codemodel.builder.java.ClassType.*

class CommonsModuleConfigurator extends JavaModuleConfigurator<CommonsModuleConfigurator> {

    static List<Integer> HTTP_STATUSES = [HttpResponse.DEFAULT] + (100..102) +
            (200..208) + [226] +
            (300..308) +
            (400..418) + (422..426) + (449..451) + [420, 428, 429, 431, 444, 499] +
            (500..511) + [598, 599]

    CommonsModuleConfigurator(CodegenParams params = new CodegenParams()) {
        super(params
                .withIfNotSet(SpringRestParams.DEFAULTS)
                .withIfNotSet(new ValidationParams()))
        generateIfUsed = true
        init()
    }

    JavaModuleBuilder makeBuilder() {
        new JavaModuleBuilder('commons', null, moduleConfig)
    }

    private void init() {
        initRestCommon()

        addFixedClass(VALIDATION_CONTEXT, 'ValidationContext')
        addFixedClass(VALIDATION_CHECKER, 'Checker', [VALIDATION_SIMPLE_CHECKER])
        addFixedClass(VALIDATION_SIMPLE_CHECKER, 'SimpleChecker')
        addFixedClass(VALIDATION_FIELD, 'Field')
        addFixedClass(VALIDATION_ERROR, 'ValidationError', new ValidationErrorBuildStrategy())
        addFixedClass(VALIDATION_RESULT, 'ValidationResult', new ValidationResultBuildStrategy())
        addFixedClass(VALIDATION_EXCEPTION, 'ValidationException', new ValidationExceptionBuildStrategy())
        addFixedClass(VALIDATION_VALIDATOR, 'Validator', new BaseValidatorBuildStrategy())
        addFixedClass(VALIDATION_COMMON_CHECKERS, 'CommonCheckers', [ENUM_VALUE_INTERFACE, RESPONSE_DTO])

        addComponent(prepareFixedClass(VALIDATION_DICTIONARY_CHECKER, 'DictionaryChecker', new DictionaryCheckerBuildStrategy()))
        addFixedClass(VALIDATION_DICTIONARY_CODES_ENUM, 'DictionaryCodes', new PlainEnumBuildStrategy())
        addFixedClass(VALIDATION_INVALID_CODES_ENUM, 'InvalidCodes', new BaseEnumBuildStrategy())

        addFixedClass(REST_REQUEST_HANDLER_TEMPLATE, 'RestRequestHandler', [VALUE_HOLDER_TEMPLATE, REST_RESPONSE_INTERFACE, VALIDATION_RESULT])

        addComponent(prepareFixedClass(VALIDATION_RESULT_MAPPER, 'ValidationResultMapper', new ValidationResultMapperBuildStrategy()))
        addFixedClass(REST_EXCEPTION_HANDLER, 'RestExceptionHandler', [VALIDATION_EXCEPTION, VALIDATION_RESULT, VALIDATION_ERROR, VALIDATION_COMMON_CHECKERS, ENUM_VALUE_INTERFACE])

        addComponent(prepareFixedClass(ENDPOINT_PROVIDER, 'EndpointProvider', new EndpointProviderBuildStrategy()))

        addFixedClass(JAXB_LOCAL_DATE_ADAPTER, 'LocalDateAdapter')
        addFixedClass(JAXB_LOCAL_DATE_TIME_ADAPTER, 'LocalDateTimeAdapter')
    }

    private void initRestCommon() {
        addClass(prepareFixedClass(REST_MAPPER, 'SerializationUtils',
                new PrivateConstructorBuildStrategy(),
                new SerializationUtilsBuildStrategy(),
                SpringDefs.MULTIPART_FILE_MAPPER))
        addFixedClass(ENUM_VALUE_INTERFACE, 'EnumValue', new EnumValueInterfaceBuilder())
        addFixedClass(VALUE_HOLDER_TEMPLATE, 'ValueHolder')
        addFixedClass(MAP_RAW_VALUE_SERIALIZER, 'MapRawValueSerializer')
        addFixedClass(REST_RESPONSE_ABSTRACT, 'RestResponseBase', new RestResponseBaseBuildStrategy())
        addFixedClass(REST_RESPONSE_INTERFACE, 'RestResponse', new RestResponseInterfaceBuildStrategy())
        addRestResponseStatusInterfaces()
    }

    private void addRestResponseStatusInterfaces() {
        def statusesModels = HTTP_STATUSES.collect {
            new ClassBuilderConfigurator.Model(model: it, name: '' + (it == HttpResponse.DEFAULT ? 'Other' : it))
        } as List<ClassBuilderConfigurator.Model>
        typeOfCode(REST_RESPONSE_STATUS_INTERFACE)
                .setNamePrefixAndSuffix('RestResponse', '')
                .addStrategy(new RestResponseStatusInterfaceBuildStrategy())
                .setPredefinedModels(statusesModels)
                .setGenerateIfUsed(true)
        typeOfCode(REST_RESPONSE_STATUS_NO_CONTENT_INTERFACE)
                .setNamePrefixAndSuffix('RestResponse', 'NoContent')
                .addStrategy(new RestResponseStatusInterfaceBuildStrategy(withContent: false))
                .setPredefinedModels(statusesModels)
                .setGenerateIfUsed(true)
    }

    CommonsModuleConfigurator setRootPackage(String rootPackage) {
        super.setRootPackage(rootPackage)
        setPackage(rootPackage + '.validator',
                VALIDATION_CONTEXT,
                VALIDATION_CHECKER,
                VALIDATION_SIMPLE_CHECKER,
                VALIDATION_FIELD,
                VALIDATION_ERROR,
                VALIDATION_RESULT,
                VALIDATION_RESULT_MAPPER,
                VALIDATION_EXCEPTION,
                VALIDATION_VALIDATOR,
                VALIDATION_DICTIONARY_CHECKER,
                VALIDATION_DICTIONARY_CODES_ENUM,
                VALIDATION_INVALID_CODES_ENUM,
                VALIDATION_COMMON_CHECKERS)
        this
    }
}
