/*
 * Copyright (c) 2021 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.base

import pl.metaprogramming.codemodel.builder.java.ClassCmBuildStrategy
import pl.metaprogramming.codemodel.model.java.EnumItemCm
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.MethodCm
import pl.metaprogramming.codemodel.model.java.ValueCm
import pl.metaprogramming.metamodel.data.EnumType

import static pl.metaprogramming.codemodel.model.java.JavaDefs.*

class BaseEnumBuildStrategy extends ClassCmBuildStrategy<EnumType> {
    static final List<Character> ALLOWED_FIRST_LETTERS = ['_' as char]

    @Override
    void makeDeclaration() {
        if (model) {
            setEnums(model.allowedValues.collect { toItem(it) })
        } else {
            classModel.isEnum = true
        }
    }

    @Override
    void makeImplementation() {
        addFields(new FieldCm(T_STRING, 'value')
                .setAnnotations([ANNOT_LB_GETTER])
                .setModifiers(MODIFIER_PRIVATE_FINAL))
        addMethods new MethodCm(
                name: classModel.className,
                modifiers: null as String,
                params: [new FieldCm(T_STRING, 'value')],
                implBody: 'this.value = value;'
        )
    }

    EnumItemCm toItem(String value) {
        new EnumItemCm(
                name: toEnumItemName(value),
                value: ValueCm.escaped(value),
                description: model.getDescription(value)
        )
    }

    static String toEnumItemName(String value) {
        StringBuilder buf = new StringBuilder()
        Character prevC = null
        for (int idx = 0; idx < value.length(); idx++) {
            def c = value.charAt(idx)
            if ((idx == 0 && !c.isLetter() && !ALLOWED_FIRST_LETTERS.contains(c))
                    || (c.isUpperCase() && prevC?.isLowerCase())) {
                buf.append('_')
            }
            if (c.isLowerCase()) {
                buf.append(c.toUpperCase())
            } else if (c.isUpperCase() || c.isDigit()) {
                buf.append(c)
            } else {
                buf.append('_')
            }
            prevC = Character.valueOf(c)
        }
        buf.toString()
    }
}
