/*
 * Copyright (c) 2018,2020 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java

import groovy.text.SimpleTemplateEngine
import pl.metaprogramming.codemodel.builder.CodeGenerationTask
import pl.metaprogramming.codemodel.formatter.JavaCodeFormatter
import pl.metaprogramming.codemodel.model.java.ClassBuilder
import pl.metaprogramming.codemodel.model.java.ClassCm
import pl.metaprogramming.codemodel.model.java.MethodCm

class ClassCmBuilder<T> extends ClassBuilder<T> {

    boolean wasMadeImplementation = false

    String modelName
    String packageName

    ClassCm classCm

    ClassCmBuilder<T> make() {
        try {
            makeDeclaration()
            makeImplementation()
            this
        } catch (RuntimeException e) {
            throw new IllegalStateException("Can't build code model for $model", e)
        }
    }

    @Override
    Object getClassType() {
        config.classType
    }

    @Override
    ClassCm getClassCd() {
        if (classCm == null) {
            classCm = new ClassCm(
                    packageName: makePackageName(),
                    className: config.classNameBuilder.make(modelName, model))
        }
        classCm
    }

    @Override
    void makeDeclaration() {
        try {
            config.strategies.each {
                it.getInstance(this).makeDeclaration()
            }
        } catch (Exception e) {
            panic(e)
        }
    }

    @Override
    void makeDecoration() {
        config.strategies.each {
            it.getInstance(this).makeDecoration()
        }
    }

    @Override
    void makeImplementation() {
        try {
            config.strategies.each {
                it.getInstance(this).makeImplementation()
            }
            wasMadeImplementation = true
        } catch (Exception e) {
            panic(e)
        }
    }

    private void panic(Exception e) {
        throw new IllegalStateException("Can't build $classCm ($config.classType for $model)", e)
    }

    @Override
    CodeGenerationTask makeGenerationTask(String filePath) {
        new CodeGenerationTask(
                destFilePath: filePath,
                codeModel: classCm,
                formatter: new JavaCodeFormatter(classCm)
        )
    }

    void addMethod(MethodCm methodCm) {
        methodCm.ownerClass = classCm
        classCm.addMethod(methodCm)
    }

    private String makePackageName() {
        if (packageName) {
            return packageName
        }
        if (!config.packageName) {
            throw new IllegalStateException("Undefined package for class type ${classType}.")
        }
        if (config.packageName.contains('$')) {
            def modelMap = model.properties
            return new SimpleTemplateEngine()
                    .createTemplate(config.packageName)
                    .make(modelMap).toString()
        }
        config.packageName
    }

    @Override
    String toString() {
        "Builder of $config.classType $modelName"
    }
}
