/*
 * Copyright (c) 2021 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codegen.spring.rs.builders

import pl.metaprogramming.codegen.spring.rs.ResponseClassResolver
import pl.metaprogramming.codemodel.builder.java.mapper.BaseMethodCmBuilder
import pl.metaprogramming.codemodel.model.java.ClassCm
import pl.metaprogramming.codemodel.model.java.MethodCm
import pl.metaprogramming.metamodel.oas.Operation
import pl.metaprogramming.metamodel.oas.Parameter

import static pl.metaprogramming.codegen.spring.rs.TypeOfCode.REQUEST_DTO
import static pl.metaprogramming.codemodel.builder.java.ClassType.ENDPOINT_PROVIDER
import static pl.metaprogramming.codemodel.builder.java.spring.SpringDefs.*
import static pl.metaprogramming.codemodel.formatter.BaseJavaCodeFormatter.NEW_LINE
import static pl.metaprogramming.codemodel.model.java.JavaDefs.*
import static pl.metaprogramming.metamodel.oas.ParamLocation.*

// see RestOutRequestMapperBuilder which generate similar code
class RestTemplateCallMethodBuilder extends BaseMethodCmBuilder<Operation> {

    static final String VAR_REQUEST = 'request'
    static final String VAR_PATH_PARAMS = 'pathParams'
    static final String VAR_HEADER_PARAMS = 'pathParams'
    static final String VAR_URL = 'uri'
    static final String VAR_REQ_ENTITY = 'entity'

    ClassCm requestClass

    RestTemplateCallMethodBuilder(Operation operation) {
        modelMapper = { operation }
    }

    @Override
    MethodCm makeDeclaration() {
        requestClass = getClass(REQUEST_DTO) as ClassCm
        new MethodCm(
                name: model.code,
                description: makeDescription(),
                resultType: ResponseClassResolver.resolve(strategy, model)
        ).addParam(VAR_REQUEST, requestClass)
    }

    String makeDescription() {
        String head = "[$model.type] $model.path"
        model.description ? "$head<br/>$NEW_LINE$model.description" : head
    }

    @Override
    String makeImplBody() {
        addPathParamsVar()
        addEndpointUrlVar()
        addHeadersVar()
        addRequestEntityVar()
        addCall()
        codeBuf.take()
    }

    void addCall() {
        if (model.successResponseSchema) {
            codeBuf
                    .addLines("return ${componentRef(REST_TEMPLATE)}.exchange($VAR_REQ_ENTITY, ${responseBodyClassValue})")
                    .add(methodCm.resultType == RESPONSE_ENTITY ? ';' : '.getBody();')
        } else {
            codeBuf.addLines("${componentRef(REST_TEMPLATE)}.exchange($VAR_REQ_ENTITY, Void.class);")
        }
    }

    void addPathParamsVar() {
        if (pathParams) {
            strategy.addImports(T_MAP, T_HASH_MAP)
            codeBuf.addLines("Map<String, Object> $VAR_PATH_PARAMS = new HashMap<>();")
            codeBuf.addLines(pathParams.collect { "${VAR_PATH_PARAMS}.put${paramNameValue(it)};" })
        }
    }

    void addEndpointUrlVar() {
        strategy.addImports(T_URI, 'org.springframework.web.util.UriComponentsBuilder')
        codeBuf
                .addLines("URI $VAR_URL = UriComponentsBuilder")
                .indentInc(2)
                .addLines(".fromUriString(${componentRef(ENDPOINT_PROVIDER)}.getEndpoint(\"$model.path\"))")
                .addLines(model.getParameters(QUERY).collect { ".queryParam${paramNameValue(it)}" })
                .addLines(pathParams ? ".build($VAR_PATH_PARAMS)" : '.build().toUri()')
                .add(';')
                .indentInc(-2)
    }

    void addRequestEntityVar() {
        strategy.addImports(REQUEST_ENTITY)
        def bodyRef = makeBodyRef()
        codeBuf.addLines("RequestEntity<?> $VAR_REQ_ENTITY = RequestEntity.${model.type.name().toLowerCase()}($VAR_URL)")
        if (headerParams) {
            codeBuf.add(".headers($VAR_HEADER_PARAMS)")
        }
        codeBuf.add(bodyRef ? ".body($bodyRef);" : '.build();')
    }

    void addHeadersVar() {
        if (headerParams) {
            addVarDeclaration(VAR_HEADER_PARAMS, HTTP_HEADERS, 'new')
            headerParams.each {
                def field = requestClass.getField(it)
                def paramRef = "${VAR_REQUEST}.get${field.name.capitalize()}()"
                if (it.isRequired) {
                    strategy.addImport(OBJECTS)
                    codeBuf.addLines("${VAR_HEADER_PARAMS}.add(\"$it.name\", Objects.requireNonNull($paramRef));")
                } else {
                    strategy.addImports(OPTIONAL)
                    codeBuf.addLines("Optional.ofNullable($paramRef).ifPresent(v -> ${VAR_HEADER_PARAMS}.add(\"$it.name\", ${transform(T_STRING, field.alias('v').setNonnull(true))}));")
                }
            }
        }
    }

    String makeBodyRef() {
        model.multipart ? addMultipartBodyVar()
                : model.requestBodySchema ? getter(VAR_REQUEST, model.requestBodySchema.code) : null
    }


    Collection<Parameter> getPathParams() {
        model.getParameters(PATH)
    }

    Collection<Parameter> getHeaderParams() {
        model.getParameters(HEADER)
    }

    String addMultipartBodyVar() {
        strategy.addImports(MULTI_VALUE_MAP,
                'org.springframework.util.LinkedMultiValueMap')
        String var = 'body'
        codeBuf.addLines("MultiValueMap<String, Object> $var = new LinkedMultiValueMap<>();")
        codeBuf.addLines(model.getParameters(FORMDATA).collect {
            "${var}.add${paramNameValue(it)};"
        })
        codeBuf.addLines(model.multipartFormDataRequestBody?.fields?.collect {
            "${var}.add(\"$it.code\", ${strategy.getter([VAR_REQUEST, it.code])});"
        })
        var
    }

    String paramNameValue(Parameter parameter) {
        "(\"$parameter.name\", ${strategy.getter([VAR_REQUEST, parameter.name])})"
    }

    private String getResponseBodyClassValue() {
        def classCd = methodCm.resultType == RESPONSE_ENTITY
                ? methodCm.resultType.genericParams[0]
                : methodCm.resultType
        if (classCd.genericParams) {
            def responseClassType = classNameFormatter.format(classCd)
            strategy.addImports('org.springframework.core.ParameterizedTypeReference')
            "new ParameterizedTypeReference<$responseClassType>() {}"
        } else {
            classNameFormatter.classRef(classCd)
        }
    }

}
