/*
 * Copyright (c) 2021 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codegen.spring.rs.builders

import groovy.transform.TypeChecked
import groovy.transform.TypeCheckingMode
import groovy.util.logging.Slf4j
import pl.metaprogramming.codemodel.builder.java.config.ValidationParams
import pl.metaprogramming.codemodel.builder.java.dto.BaseDtoBuildStrategy
import pl.metaprogramming.codemodel.builder.java.spring.mapper.DefaultValueMapper
import pl.metaprogramming.codemodel.model.java.AnnotationCm
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.ValueCm
import pl.metaprogramming.metamodel.data.DataSchema
import pl.metaprogramming.metamodel.oas.Parameter

import static pl.metaprogramming.codemodel.model.java.JavaDefs.*
import static pl.metaprogramming.codegen.spring.rs.TypeOfCode.DTO
import static pl.metaprogramming.codemodel.builder.java.MetaModelAttribute.JAVA_NAME
import static pl.metaprogramming.codemodel.formatter.BaseJavaCodeFormatter.NEW_LINE

@Slf4j
class DtoBuildStrategy extends BaseDtoBuildStrategy {

    JakartaBeanValidationSupport jbvSupport = new JakartaBeanValidationSupport()
    boolean withDescriptiveFields = true
    boolean withJsonAnnotations = true

    @Override
    void addField(DataSchema schema) {
        def fieldType = getClass(DTO, schema.dataType)
        FieldCm fieldCm = new FieldCm(
                model: schema,
                name: schema.getAdditive(JAVA_NAME),
                description: schema.description,
                type: fieldType,
                value: schema.defaultValue ? DefaultValueMapper.make(fieldType, schema.defaultValue, this) : null,
                annotations: makeFieldAnnotations(schema))
        if (shouldAddDescriptiveFields()) {
            addDescriptiveFields(schema, fieldCm)
        }
        if (schema instanceof Parameter) {
            String paramDesc = "$schema.name $schema.location parameter."
            fieldCm.description = fieldCm.description
                    ? paramDesc + NEW_LINE + '<br/>' + fieldCm.description
                    : paramDesc
        }
        addFields(fieldCm)
    }

    private List<AnnotationCm> makeFieldAnnotations(DataSchema schema) {
        def result = []
        if (schema.code != schema.getAdditive(JAVA_NAME) && withJsonAnnotations) {
            result.add(new AnnotationCm('com.fasterxml.jackson.annotation.JsonProperty',
                    [value: ValueCm.escaped(schema.code)]))
        }
        if (isJbvEnabled()) {
            jbvSupport.addAnnotations(result as List<AnnotationCm>, schema)
        } else {
            result.add(schema.isRequired ? ANNOT_NON_NULL : ANNOT_NULLABLE)
        }
        result
    }

    @TypeChecked(TypeCheckingMode.SKIP)
    private boolean isJbvEnabled() {
        getParams(ValidationParams).useJakartaBeanValidation
    }

    private boolean shouldAddDescriptiveFields() {
        withDescriptiveFields && !isJbvEnabled()
    }
}
