/*
 * Copyright (c) 2021 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codegen.spring.rs.builders

import groovy.transform.TypeChecked
import groovy.transform.TypeCheckingMode
import pl.metaprogramming.codegen.spring.rs.ResponseClassResolver
import pl.metaprogramming.codemodel.builder.java.ClassCmBuildStrategy
import pl.metaprogramming.codemodel.builder.java.config.SpringRestParams
import pl.metaprogramming.codemodel.builder.java.config.ValidationParams
import pl.metaprogramming.codemodel.builder.java.spring.RestParamsBuilder
import pl.metaprogramming.codemodel.builder.java.spring.mapper.DefaultValueMapper
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.JavaDefs
import pl.metaprogramming.codemodel.model.java.MethodCm
import pl.metaprogramming.metamodel.data.DataSchema
import pl.metaprogramming.metamodel.oas.Operation

import static pl.metaprogramming.codegen.spring.rs.TypeOfCode.DTO
import static pl.metaprogramming.codegen.spring.rs.TypeOfCode.REQUEST_DTO
import static pl.metaprogramming.codemodel.builder.java.ClassType.VALIDATION_RESULT
import static pl.metaprogramming.codemodel.builder.java.spring.SpringDefs.*

class ControllerBuildStrategy extends ClassCmBuildStrategy<Operation> {
    @Override
    void makeImplementation() {
        addAnnotation(ANNOT_REST_CONTROLLER)
        addMethods makeHandleMethod()
        // force DTO generation for non successful responses
        model.responses.each {
            if (it.schema?.object)  getClass(DTO, it.schema.objectType)
        }
    }

    private MethodCm makeHandleMethod() {
        def methodCm = new RestParamsBuilder(this, DTO)
                .setUseJakartaBeanValidation(isJbvEnabled())
                .setUseDataAutoconversion(true)
                .makeControllerMethod(model)
        def responseClass = ResponseClassResolver.resolve(this, model)
        def request = makeRequestField(methodCm.params + additionalFields)
        if (!isJbvEnabled()) {
            def validationCall = transform(request, getClass(VALIDATION_RESULT))
            codeBuf.addLines("${validationCall.value};")
        }
        def serviceCall = transform(request, responseClass)
        if (responseClass.isVoid()) {
            methodCm.resultType = methodCm.resultType.withGeneric(ClassCd.VOID)
            methodCm.implBody = codeBuf.addLines(
                    "$serviceCall.expression;",
                    "return ${emptyResponse};"
            ).take()
        } else if (responseClass.isClass(RESPONSE_ENTITY)) {
            methodCm.resultType = responseClass
            methodCm.implBody = codeBuf.addLines(
                    "return ${serviceCall.expression};"
            ).take()
        } else {
            methodCm.resultType = methodCm.resultType.withGeneric(responseClass)
            methodCm.implBody = codeBuf.addLines(
                    "return ${getResponse(serviceCall.expression)};"
            ).take()
        }
        methodCm
    }

    private FieldCm makeRequestField(List<FieldCm> fields) {
        def requestClass = getClass(REQUEST_DTO)
        addImport(requestClass)
        codeBuf.newLine("$requestClass.className request = new $requestClass.className()")
        codeBuf.indent(2)
        fields.each {
            codeBuf.newLine(".set${it.name.capitalize()}(${getParamValue(it)})")
        }
        codeBuf.add(';').indent()
        new FieldCm(requestClass, 'request')
    }


    @TypeChecked(TypeCheckingMode.SKIP)
    private List<FieldCm> getAdditionalFields() {
        getParams(SpringRestParams).injectBeansIntoRequest
    }


    private String getParamValue(FieldCm param) {
        def schema = param.model as DataSchema
        if (schema?.defaultValue) {
            addImports(JavaDefs.OPTIONAL)
            "Optional.ofNullable($param.name).orElse(${DefaultValueMapper.make(param.type, schema.defaultValue, this)})"
        } else if (param.type.isClass(MULTIPART_FILE)) {
            transform(param, JavaDefs.T_BYTE_ARRAY).value.toString()
        } else {
            param.name
        }
    }

    private String getEmptyResponse() {
        def statusMethod = model.successResponse
                ? STATUS_2_METHOD.get(model.successResponse.status)
                : 'noContent'
        def responseBuildExp = statusMethod ? "${statusMethod}()" : "status(${model.successResponse.status})"
        "ResponseEntity.${responseBuildExp}.build()"
    }

    private String getResponse(String serviceCallExp) {
        def statusMethod = STATUS_2_METHOD.get(model.successResponse.status)
        def responseBuildExp = statusMethod ? "${statusMethod}($serviceCallExp)"
                : "status(${model.successResponse.status}).body($serviceCallExp)"
        "ResponseEntity.${responseBuildExp}"
    }

    @TypeChecked(TypeCheckingMode.SKIP)
    private boolean isJbvEnabled() {
        getParams(ValidationParams).useJakartaBeanValidation
    }

    private static Map<Integer, String> STATUS_2_METHOD = [
            200: 'ok',
            204: 'noContent',
    ]
}
