/*
 * Copyright (c) 2021 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codegen.spring.rs

import groovy.transform.TypeChecked
import groovy.transform.TypeCheckingMode
import pl.metaprogramming.codemodel.builder.java.config.JavaModuleConfig
import pl.metaprogramming.codemodel.builder.java.config.SpringRestParams
import pl.metaprogramming.codemodel.builder.java.config.ValidationParams
import pl.metaprogramming.codemodel.builder.java.module.JavaModuleBuilder
import pl.metaprogramming.codemodel.formatter.JavaCodeFormatter
import pl.metaprogramming.metamodel.oas.RestApi

import static pl.metaprogramming.codegen.spring.rs.TypeOfCode.*
import static pl.metaprogramming.codemodel.builder.java.ClassType.REST_EXCEPTION_HANDLER

class SpringRestServicesBuilder extends JavaModuleBuilder<RestApi> {

    SpringRestServicesBuilder(String moduleName, RestApi moduleModel, JavaModuleConfig config) {
        super(moduleName, moduleModel, config, CLASS_TYPE_BUILD_ORDER)
        if (generateValidator()) {
            operationClassTypes.add(REQUEST_VALIDATOR)
        }
        if (!delegateToFacade()) {
            operationClassTypes.addAll(OPERATION_EXECUTOR, OPERATION_EXECUTOR_IMPL)
        }
    }

    @Override
    void make() {
        addSchemaCodes(model)
        addOperationCodes(model)
        if (generateValidator()) {
            classIndex.useClass(REST_EXCEPTION_HANDLER)
        }
        makeCodeModels()
    }

    void addSchemaCodes(RestApi model) {
        model.schemas.each { schema ->
            if (schema.isEnum()) {
                addClass(ENUM, schema)
            }
            if (schema.isObject()) {
                addClass(DTO, schema)
                if (generateValidator()) {
                    addClass(DTO_VALIDATOR, schema)
                }
            }
        }
    }

    void addOperationCodes(RestApi model) {
        model.groupedOperations.each { group, operations ->
            String javaGroupName = JavaCodeFormatter.toCapitalizeJavaName(group)
            operations.each { operation ->
                String dataName = operation.code.capitalize()
                operationClassTypes.each { classType ->
                    addClass(classType, operation, dataName)
                }
            }
            if (delegateToFacade()) {
                [FACADE_IMPL, FACADE].each { classType ->
                    addClass(classType, operations, javaGroupName)
                }
            }
        }
    }

    @TypeChecked(TypeCheckingMode.SKIP)
    private boolean generateValidator() {
        !getParams(ValidationParams).useJakartaBeanValidation
    }

    @TypeChecked(TypeCheckingMode.SKIP)
    private boolean delegateToFacade() {
        getParams(SpringRestParams).delegateToFacade
    }

    private final List operationClassTypes = [
            REQUEST_DTO,
            REST_CONTROLLER]

    private static final List CLASS_TYPE_BUILD_ORDER = [
            ENUM,
            DTO,
            REQUEST_DTO,
            FACADE_IMPL, FACADE,
            OPERATION_EXECUTOR_IMPL, OPERATION_EXECUTOR,
            DTO_VALIDATOR,
            REQUEST_VALIDATOR,
            REST_CONTROLLER
    ]
}
