/*
 * Copyright (c) 2021 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codegen.spring

import pl.metaprogramming.codegen.spring.rs.SpringRestServicesBuilder
import pl.metaprogramming.codegen.spring.rs.builders.*
import pl.metaprogramming.codemodel.builder.java.ClassNameBuilder
import pl.metaprogramming.codemodel.builder.java.base.InterfaceBuildStrategy
import pl.metaprogramming.codemodel.builder.java.config.*
import pl.metaprogramming.codemodel.builder.java.rest.RestDtoValidatorBuildStrategy
import pl.metaprogramming.codemodel.builder.java.rest.RestRequestValidatorBuildStrategy
import pl.metaprogramming.metamodel.data.DataType
import pl.metaprogramming.metamodel.oas.Operation
import pl.metaprogramming.metamodel.oas.OperationType
import pl.metaprogramming.metamodel.oas.RestApi

import static pl.metaprogramming.codegen.spring.rs.TypeOfCode.*
import static pl.metaprogramming.codemodel.model.java.JavaDefs.T_ZONED_DATE_TIME

class SpringRestServicesConfigurator extends JavaModuleConfigurator<SpringRestServicesConfigurator> {

    SpringRestServicesConfigurator(CodegenParams params = new CodegenParams()) {
        super(params.withIfNotSet(new ValidationParams()).withIfNotSet(new SpringRestParams()))
        dataTypeMapper.setMapping(DataType.DATE_TIME, T_ZONED_DATE_TIME)
        generateIfUsed = true
        init()
    }

    SpringRestServicesBuilder makeBuilder(String moduleName, RestApi moduleModel) {
        new SpringRestServicesBuilder(moduleName, moduleModel, moduleConfig)
    }

    private void init() {
        params.get(ValidationParams).throwExceptionIfValidationFailed = true
        def additionalBeansForRequestBuildStrategy = new AdditionalFieldsBuildStrategy(
                params.get(SpringRestParams).injectBeansIntoRequest)
        typeOfCode(REQUEST_VALIDATOR)
                .setNameSuffix('Validator')
                .addStrategy(componentStrategy, diStrategy, new RestRequestValidatorBuildStrategy(
                        dtoTypeOfCode: REQUEST_DTO, validatorTypeOfCode: DTO_VALIDATOR, enumTypeOfCode: DTO
                ))
        typeOfCode(DTO_VALIDATOR)
                .setNameSuffix('Validator')
                .addStrategy(componentStrategy, diStrategy, new RestDtoValidatorBuildStrategy(
                        dtoTypeOfCode: DTO, validatorTypeOfCode: DTO_VALIDATOR, enumTypeOfCode: DTO
                ))
        typeOfCode(ENUM)
                .addStrategy(new EnumBuildStrategy())
                .setRegisterClassType(DTO)
        typeOfCode(DTO)
                .setNameSuffix('Dto')
                .addStrategy(new DtoBuildStrategy(), LOMBOK_DATA_STRATEGY)
        typeOfCode(REQUEST_DTO)
                .setNameSuffix('Request')
                .addStrategy(additionalBeansForRequestBuildStrategy, new DtoBuildStrategy(modelMapper: SpringRestInAdapterConfigurator.OPERATION_REQUEST_SCHEMA), LOMBOK_DATA_STRATEGY)
        typeOfCode(REST_CONTROLLER)
                .setNameSuffix('Controller')
                .addStrategy(new ControllerBuildStrategy(), additionalBeansForRequestBuildStrategy, diStrategy)
        typeOfCode(OPERATION_EXECUTOR_IMPL)
                .setClassNameBuilder(this::operationExecutorClassNameBuilder as ClassNameBuilder<Operation>)
                .addStrategy(new OperationExecutorBuildStrategy())
        typeOfCode(OPERATION_EXECUTOR)
                .setClassNameBuilder(this::operationExecutorInterfaceNameBuilder as ClassNameBuilder<Operation>)
                .addStrategy(new InterfaceBuildStrategy(OPERATION_EXECUTOR_IMPL))
        typeOfCode(FACADE_IMPL)
                .setNameSuffix('FacadeImpl')
                .addStrategy(new FacadeBuildStrategy())
        typeOfCode(FACADE)
                .setNameSuffix('Facade')
                .addStrategy(new InterfaceBuildStrategy(FACADE_IMPL))

        generateAlways(FACADE, FACADE_IMPL, OPERATION_EXECUTOR, OPERATION_EXECUTOR_IMPL, REST_CONTROLLER)
    }

    @Override
    SpringRestServicesConfigurator setRootPackage(String rootPackage) {
        setPackages(rootPackage, 'rest')
    }

    SpringRestServicesConfigurator setPackages(String rootPackage, String adapterName) {
        super.setRootPackage(rootPackage)
        setPackage("${rootPackage}.process", OPERATION_EXECUTOR_IMPL)
        setPackage("${rootPackage}.ports.in.${adapterName}", FACADE, OPERATION_EXECUTOR)
        setPackage("${rootPackage}.ports.in.${adapterName}.dtos", DTO, ENUM, REQUEST_DTO)
        setPackage("${rootPackage}.adapters.in.${adapterName}.controllers", REST_CONTROLLER)
        setPackage("${rootPackage}.adapters.in.${adapterName}.validators", REQUEST_VALIDATOR, DTO_VALIDATOR)
        this
    }

    String operationExecutorInterfaceNameBuilder(String modelName, Operation operation) {
        'I' + operationExecutorClassNameBuilder(modelName, operation)
    }

    String operationExecutorClassNameBuilder(String modelName, Operation operation) {
        modelName.capitalize() +
                (OperationType.GET == operation.type ? 'Query' : 'Command')
    }
}
