/*
 * Copyright (c) 2021 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codegen.spring

import pl.metaprogramming.codegen.spring.rs.SpringRestClientsBuilder
import pl.metaprogramming.codegen.spring.rs.builders.ClientBuildStrategy
import pl.metaprogramming.codegen.spring.rs.builders.DtoBuildStrategy
import pl.metaprogramming.codegen.spring.rs.builders.EnumBuildStrategy
import pl.metaprogramming.codemodel.builder.java.base.InterfaceBuildStrategy
import pl.metaprogramming.codemodel.builder.java.config.*
import pl.metaprogramming.codemodel.builder.java.spring.SpringDefs
import pl.metaprogramming.metamodel.data.DataType
import pl.metaprogramming.metamodel.oas.RestApi

import static pl.metaprogramming.codegen.spring.rs.TypeOfCode.*
import static pl.metaprogramming.codemodel.model.java.JavaDefs.T_ZONED_DATE_TIME

class SpringRestClientsConfigurator extends JavaModuleConfigurator<SpringRestClientsConfigurator> {

    SpringRestClientsConfigurator(CodegenParams params = new CodegenParams()) {
        super(params.withIfNotSet(new ValidationParams()).withIfNotSet(new SpringRestParams()))
        dataTypeMapper.setMapping(DataType.DATE_TIME, T_ZONED_DATE_TIME)
        dataTypeMapper.setMapping(DataType.BINARY, SpringDefs.RESOURCE)
        generateIfUsed = true
        init()
    }


    SpringRestClientsBuilder makeBuilder(String moduleName, RestApi moduleModel) {
        new SpringRestClientsBuilder(moduleName, moduleModel, moduleConfig)
    }

    private void init() {
        typeOfCode(ENUM)
                .addStrategy(new EnumBuildStrategy())
                .setRegisterClassType(DTO)
        typeOfCode(DTO)
                .setNameSuffix('Dto')
                .addStrategy(new DtoBuildStrategy(withDescriptiveFields: false), LOMBOK_DATA_STRATEGY)
        typeOfCode(REQUEST_DTO)
                .setNameSuffix('Request')
                .addStrategy(LOMBOK_DATA_STRATEGY, new DtoBuildStrategy(
                        modelMapper: SpringRestInAdapterConfigurator.OPERATION_REQUEST_SCHEMA,
                        withDescriptiveFields: false,
                        withJsonAnnotations: false
                ))
        typeOfCode(CLIENT_IMPL)
                .setNameSuffix('ClientImpl')
                .addStrategy(new ClientBuildStrategy(), componentStrategy, diStrategy)
        typeOfCode(CLIENT)
                .setNameSuffix('Client')
                .addStrategy(new InterfaceBuildStrategy(CLIENT_IMPL))

        generateAlways(CLIENT_IMPL, CLIENT)
    }


    @Override
    SpringRestClientsConfigurator setRootPackage(String rootPackage) {
        setPackages(rootPackage, 'rest')
    }


    SpringRestClientsConfigurator setPackages(String rootPackage, String adapterName) {
        super.setRootPackage(rootPackage)
        setPackage("${rootPackage}.ports.out.${adapterName}.dtos", DTO, ENUM, REQUEST_DTO)
        setPackage("${rootPackage}.adapters.out.${adapterName}", CLIENT, CLIENT_IMPL)
        this
    }

}
