/*
 * Copyright (c) 2021 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codegen.generator

import groovy.transform.ToString
import org.yaml.snakeyaml.DumperOptions
import org.yaml.snakeyaml.DumperOptions.LineBreak
import org.yaml.snakeyaml.Yaml
import pl.metaprogramming.codemodel.formatter.JavaCodeFormatter

import java.nio.file.Files
import java.util.function.Predicate

class CodeGenerator {
    static String DATETIME_FORMAT = 'yyyy-MM-dd HH:mm:ss'

    CodegenConfig cfg
    Map<String, FileMetadata> fileIndex = [:]
    String prevGenerationDate
    String generationDate

    static CodeGenerator run(CodegenConfig cfg) {
        new CodeGenerator(cfg).runAll()
    }

    static CodeGenerator cleanup(CodegenConfig cfg, boolean total) {
        new CodeGenerator(cfg).cleanup(total)
    }

    CodeGenerator(CodegenConfig cfg) {
        this.cfg = cfg
    }

    CodeGenerator runAll() {
        loadIndex(FileStatus.ABANDONED)
        generationDate = new Date().format(DATETIME_FORMAT)
        cfg.modules.each { it.make() }
        cfg.modules.each {
            it.codesToGenerate.each {
                generateFile(it.destFilePath, it.content)
            }
        }
        removeFiles('Delete abandoned files') {
            it.status == FileStatus.ABANDONED
        }
        saveIndex()
        this
    }

    CodeGenerator cleanup(boolean total) {
        loadIndex(FileStatus.UNCHANGED)
        def removed = removeFiles("Delete generated files [${total ? 'include' : 'skip'} manually modified codes]") {
            total || it.status != FileStatus.DETACHED
        }
        fileIndex.removeAll { removed.contains(it.value) }
        Files.walk(cfg.baseDir.toPath())
                .sorted(Comparator.reverseOrder())
                .map({ it.toFile() })
                .filter({ it.isDirectory() })
                .forEach({ it.delete() })
        saveIndex()
        this
    }

    private void generateFile(String relativeFilePath, String content) {
        def md5 = HashBuilder.build(content)
        if (fileIndex.containsKey(relativeFilePath)) {
            updateFileExistsInIndex(relativeFilePath, content, md5)
        } else {
            def state = getFileState(relativeFilePath)
            if (state.exists) {
                if (!state.annotatedByGenerated) {
                    if (existsIndexFile()) {
                        log "$relativeFilePath already exists, but does not have Generated annotaion"
                    }
                    addToIndex(relativeFilePath, md5, FileStatus.DETACHED)
                } else if (state.md5 == md5) {
                    if (existsIndexFile()) {
                        log "$relativeFilePath already exists, but has the same content like generated one"
                    }
                    createFile(relativeFilePath, content, md5, FileStatus.UNCHANGED)
                } else {
                    if (cfg.forceMode) {
                        log "$relativeFilePath already exists, but has other content than generated one."
                        createFile(relativeFilePath, content, md5, FileStatus.UPDATED)
                    } else {
                        throw new RuntimeException("The file $relativeFilePath already exists and have different content. Set foreceMode to proceed.")
                    }
                }
            } else {
                createFile(relativeFilePath, content, md5)
            }
        }
    }

    private void updateFileExistsInIndex(String relativeFilePath, String newContent, String newMd5) {
        def fileMetadata = fileIndex.get(relativeFilePath)
        checkFileStatus(fileMetadata)
        if (fileMetadata.state.exists && !fileMetadata.state.annotatedByGenerated) {
            fileMetadata.status = FileStatus.DETACHED
        } else if (fileMetadata.md5 == newMd5) {
            fileMetadata.status = FileStatus.UNCHANGED
        } else {
            fileMetadata.status = FileStatus.UPDATED
        }
        fileMetadata.md5 = newMd5
        log "$relativeFilePath ... ${fileMetadata.status}"
        if (fileMetadata.status == FileStatus.UPDATED
                || fileMetadata.status == FileStatus.UNCHANGED && fileMetadata.state.md5 != newMd5
        ) {
            overwrite(relativeFilePath, newContent)
        }
    }

    private void createFile(String relativeFilePath, String content, String md5, FileStatus status = FileStatus.CREATED) {
        addToIndex(relativeFilePath, md5, status)
        overwrite(relativeFilePath, content)
    }

    private void addToIndex(String relativeFilePath, String md5, FileStatus status) {
        def fileMetadata = new FileMetadata(
                path: relativeFilePath,
                md5: md5,
                status: status,
                state: new FileState(exists: false)
        )
        log "$relativeFilePath ... ${fileMetadata.status}"
        fileIndex.put(relativeFilePath, fileMetadata)
    }

    private void checkFileStatus(FileMetadata fileMetadata) {
        if (!fileMetadata.state.exists) {
            if (cfg.forceMode) {
                log "${fileMetadata.path} has been restored."
            } else {
                throw new RuntimeException("The file ${fileMetadata.path} has been manually deleted. Set foreceMode to proceed.")
            }
        }
        if (fileMetadata.status == FileStatus.MALFORMED) {
            if (cfg.forceMode) {
                log "${fileMetadata.path} has been overwritten."
            } else {
                throw new RuntimeException("The file ${fileMetadata.path} has been manually modified. Set foreceMode to proceed.")
            }
        }
//        throw new RuntimeException("The file $relativeFilePath has changes. Set foreceMode to proceed.")
    }

    private Collection<FileMetadata> removeFiles(String label, Predicate<FileMetadata> filter) {
        def filtered = fileIndex.values().findAll { filter.test(it) }
        if (filtered && label) {
            log label
        }
        filtered.each {
            log "$it.path ... $it.status"
            def file = getFile(it.path)
            if (file.exists() && !file.delete()) {
                throw new RuntimeException("Can't remove file $it")
            }
        }
        filtered
    }

    private void saveIndex() {
        Yaml yaml = new Yaml(new DumperOptions(
                defaultFlowStyle: DumperOptions.FlowStyle.BLOCK,
                lineBreak: lineBreakByCodeFormatter()
        ))
        def indexValue = [:]
        if (cfg.addLastGenerationTag) {
            indexValue.put('lastGeneration', generationDate)
        }
        indexValue.put('files', fileIndex.values()
                .findAll { it.status != FileStatus.ABANDONED }
                .sort { it.path }
                .collect {
                    [
                            path      : it.path,
                            status    : cfg.storeAnyKindOfStatusesInIndexFile || [FileStatus.DETACHED].contains(it.status) ? it.status.name() : null,
                            lastUpdate: !cfg.addLastUpdateTag ? null : [FileStatus.CREATED, FileStatus.UPDATED].contains(it.status) ? generationDate : it.lastUpdate ?: 'unknown',
                            md5       : cfg.forceMode ? null : it.md5,
                    ].findAll { it.value != null }
                })
        overwrite(cfg.indexFile, yaml.dump(indexValue))
    }

    private LineBreak lineBreakByCodeFormatter() {
        for (def lb : LineBreak.values()) {
            if (JavaCodeFormatter.NEW_LINE == lb.getString()) {
                return lb
            }
        }
        LineBreak.getPlatformLineBreak()
    }

    private void loadIndex(FileStatus defaultStatus) {
        if (!existsIndexFile()) {
            log "Index file ($cfg.indexFile.path) does not exists"
            return
        }
        Map index = new Yaml().load(cfg.indexFile.text)
        prevGenerationDate = index.lastGeneration
        index.files.each { Map<String, String> info ->
            def state = getFileState(info.path)
            def oldMd5 = cfg.forceMode ? state.md5 : info.md5
            // path auto-fix, codegen version before 0.6.1 had a bug and sometimes the relative path started with '/'
            // TODO [2021-07-19] this fix should be removed in some time
            def path = info.path.startsWith('/') ? info.path.substring(1) : info.path
            fileIndex.put(path, new FileMetadata(
                    path: path,
                    md5: oldMd5,
                    lastUpdate: info.lastUpdate,
                    state: state,
                    status: !state.annotatedByGenerated ? FileStatus.DETACHED
                            : state.md5 != oldMd5 ? FileStatus.MALFORMED
                            : defaultStatus,
            ))
        }
    }

    private boolean existsIndexFile() {
        cfg.indexFile.exists()
    }

    private FileState getFileState(String relativeFilePath) {
        def file = getFile(relativeFilePath)
        boolean isJavaOrGroovy = file.name.endsWith('.java') || file.name.endsWith('.groovy')
        HashBuilder hashBuilder = new HashBuilder()
        FileState result = new FileState()
        result.exists = file.exists()
        if (file.exists()) {
            file.eachLine {
                if (isJavaOrGroovy && !result.annotatedByGenerated) {
                    def trimmed = it.trim()
                    if (trimmed.startsWith('@Generated')
                            || trimmed.startsWith('@javax.annotation.processing.Generated')
                            || trimmed.startsWith('@javax.annotation.Generated')) {
                        result.annotatedByGenerated = true
                    }
                }
                hashBuilder.addLine(it)
            }
        }
        result.md5 = hashBuilder.hash
        result
    }

    private File getFile(String relativeFilePath) {
        new File(cfg.baseDir, relativeFilePath)
    }

    private File overwrite(String relativeFilePath, String content) {
        overwrite(new File(cfg.baseDir, relativeFilePath), content)
    }

    private File overwrite(File file, String content) {
        file.delete()
        file.parentFile.mkdirs()
        file.createNewFile()
        file.write(content, cfg.charset)
        file
    }

    @ToString
    static class FileMetadata {
        String path
        String md5
        String lastUpdate
        FileState state
        FileStatus status
    }

    @ToString
    static class FileState {
        boolean exists
        boolean annotatedByGenerated
        String md5
    }

    static enum FileStatus {
        CREATED,    // plik został utworzony przez generator
        UPDATED,    // plik został nadpisany przez generator
        UNCHANGED,  // plik nie uległ zmianie w wyniku uruchomienia generatora
        DETACHED,   // plik nie będzie nadpisywany przez generator (w java usunięcie adnotacji @Generated)
        ABANDONED,  // plik był, ale już nie jest generowany - do usunięcia
        MALFORMED,
        // suma kontrolna istniejącego pliku nie zgadza się z sumą kontrolną zapisaną w indeksie, przy czym plik nie został oznaczony jako DETACHED
    }

    private void log(String message) {
        if (cfg.verbose) {
            println message
        }
    }

}
