/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.metamodel.parser.wsdl

import com.predic8.schema.*
import com.predic8.wsdl.AbstractPortTypeMessage
import com.predic8.wsdl.Definitions
import com.predic8.wsdl.PortType
import com.predic8.wsdl.WSDLParser
import groovy.xml.QName
import pl.metaprogramming.metamodel.model.data.ArrayType
import pl.metaprogramming.metamodel.model.data.DataSchema
import pl.metaprogramming.metamodel.model.data.DataType
import pl.metaprogramming.metamodel.model.data.EnumType
import pl.metaprogramming.metamodel.model.data.ObjectType
import pl.metaprogramming.metamodel.model.wsdl.WsdlApi
import pl.metaprogramming.metamodel.model.wsdl.WsdlOperation

class WsdlParser {
    //https://www.membrane-soa.org/soa-model-doc/1.4/java-api/parse-wsdl-java-api.htm

    static String UNBOUNDED = 'unbounded'

    WsdlApi api = new WsdlApi()
    Map<String, DataSchema> allSchemas = [:]

    XsTypeParser xsTypeParser = new XsTypeParser()
    List<DataTypeParser<SimpleType>> simpleTypeParsers = [
            new SimpleTypeEnumParser(),
            new SimpleTypeBaseParser()
    ]

    WsdlApi parse(InputStream is) {
        WSDLParser parser = new WSDLParser()
        Definitions defs = parser.parse(is)
        parseDataSchemas(defs.schemas)
        resolveUnknownDataTypes(api.schemas.values())
        parseOperations(defs.portTypes)
        assert defs.services.size() == 1
        api.name = defs.services[0].name
        api.uri = defs.services[0].ports[0].address.location
        api
    }

    private parseOperations(List<PortType> portTypes) {
        portTypes.each { pt ->
            pt.operations.each { op ->
                api.operations.add(new WsdlOperation(
                        name: op.name,
                        input: toDataSchema(op.input),
                        output: toDataSchema(op.output)
                ))
            }
        }
    }

    private DataSchema toDataSchema(AbstractPortTypeMessage message) {
        def qname = toQname(message)
        def findMessage = message.definitions.messages.find { it.qname == qname }
        assert findMessage.parts.size() == 1
        def result = toDataSchema(findMessage.parts[0].element.qname)
        result.objectType.isRootElement = true
        result
    }

    static private QName toQname(AbstractPortTypeMessage message) {
        new QName(
                message.getNamespace(message.messagePrefixedName.prefix) as String,
                message.messagePrefixedName.localName,
                message.messagePrefixedName.prefix)
    }

    private parseDataSchemas(List<Schema> schemas) {
        schemas.each {
            api.namespaceElementFormDefault.put(it.targetNamespace, it.elementFormDefault)
            it.complexTypes.each { addSchema(it) }
            it.simpleTypes.each { addSchema(it) }
            it.allElements.each { addSchema(it) }
        }
    }

    private void addSchema(ComplexType component) {
        addSchema(component, toObject(component.model, component.name))
    }

    private void addSchema(Element component) {
        if (component.embeddedType instanceof ComplexType) {
            addSchema(component, toObject((component.embeddedType as ComplexType).model, component.name))
        } else if (component.type) {
            addSchema(component, toDataType(component.type), false)
        } else {
            throw new RuntimeException("Can't handle $component")
        }
    }

    private void addSchema(SimpleType component) {
        def dataType = simpleTypeParsers.findResult { it.parse(component) }
        assert dataType, "Can't handle $component"
        addSchema(component, dataType, dataType instanceof EnumType)
    }

    private void addSchema(SchemaComponent component, DataType dataType, boolean addToApi = true) {
        def dataSchema = new DataSchema(
                code: component.name,
                namespace: component.namespaceUri,
                dataType: dataType,
        )
        allSchemas.put(dataSchema.qname, dataSchema)
        if (addToApi) {
            api.schemas.put(dataSchema.qname, dataSchema)
        }
    }

    private ObjectType toObject(SchemaComponent component, String code) {
        if (component instanceof Sequence) {
            return new ObjectType(
                    code: code,
                    fields: component.elements.collect {
                        def dataType = toDataType(it.type)
                        isList(it) ?
                                new DataSchema(
                                        code: it.name,
                                        dataType: new ArrayType(
                                                itemsSchema: new DataSchema(dataType: dataType),
                                                minItems: Integer.valueOf(it.minOccurs),
                                                maxItems: it.maxOccurs != UNBOUNDED ? Integer.valueOf(it.maxOccurs) : null,
                                        ),
                                        isNillable: it.nillable,
                                )
                                :
                                new DataSchema(
                                        code: it.name,
                                        namespace: it.type.namespaceURI,
                                        dataType: dataType,
                                        isRequired: it.minOccurs != '0',
                                        isNillable: it.nillable,
                                )
                    })
        }
        throw new RuntimeException("Can't handle: $code, $component")
    }

    private static boolean isList(Element it) {
        it.maxOccurs == UNBOUNDED || Integer.parseInt(it.maxOccurs) > 1
    }

    private DataSchema toDataSchema(QName qname, String code = null) {
        new DataSchema(
                code: code ?: qname.localPart,
                namespace: qname.namespaceURI,
                dataType: toDataType(qname, true)
        )
    }

    private DataType toDataType(QName qName, boolean required = false) {
        xsTypeParser.parse(qName) ?: findDataType(qName, required)
    }

    private void resolveUnknownDataTypes(Collection<DataSchema> dataSchemas) {
        dataSchemas.each { resolveUnknownDataType(it) }
    }

    void resolveUnknownDataType(DataSchema dataSchema) {
        if (dataSchema.dataType instanceof UnknownDataType) {
            dataSchema.dataType = findDataType((dataSchema.dataType as UnknownDataType).qname, true)
        }
        if (dataSchema.object) {
            resolveUnknownDataTypes(dataSchema.objectType.fields)
        }
        if (dataSchema.array) {
            resolveUnknownDataType(dataSchema.arrayType.itemsSchema)
        }
    }

    DataType findDataType(QName qname, boolean required = false) {
        if (allSchemas.containsKey(qname.toString())) {
            return allSchemas[qname.toString()].dataType
        }
        if (required) {
            throw new RuntimeException("Unknown data type: " + qname)
        }
        new UnknownDataType(qname)

    }
}
