/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.metamodel.parser.swagger

import pl.metaprogramming.metamodel.RestApiMetaModelChecker
import pl.metaprogramming.metamodel.model.data.DataType
import pl.metaprogramming.metamodel.model.rest.HttpResponse
import pl.metaprogramming.metamodel.model.rest.Parameter
import pl.metaprogramming.metamodel.model.rest.enums.OperationType

class OperationsParser extends BaseParser {

    static final String CONTROLLER_TAG = 'x-swagger-router-controller'
    static final List OPERATION_TYPES = OperationType.values().collect { it.name().toLowerCase() }
    static final List NO_OPERATION_PATH_ELEMENTS = [CONTROLLER_TAG]

    DefinitionsParser definitionsParser
    ParametersParser parametersParser

    static class SwaggerOperation {
        String path
        String operationType
        String operationId
        String controller
        Map spec
    }

    OperationsParser(BaseParser template, DefinitionsParser definitionsParser, ParametersParser parametersParser) {
        configure(template)
        this.parametersParser = parametersParser
        this.definitionsParser = definitionsParser
    }

    void readOperations() {
        log "Going to parse operations..."
        getOperations().each { operation ->
            log "  $operation.operationId"
            builder.operation(
                    operation.controller,
                    operation.operationId,
                    operation.path,
                    toOperationType(operation.operationType),
                    (String) operation.spec['summary'],
                    {
                        it.requestBody = getOperationRequestBody(operation.spec)
                        it.parameters = getOperationParameters(operation.spec) + config.additionalParameters
                        it.responses = getOperationResponses(operation.spec)
                        it.consumes = getOperationConsumes(operation)
                        it.produces = getOperationProduces(operation)
                    })
        }

    }


    static OperationType toOperationType(String code) {
        OperationType.valueOf(OperationType, code.toUpperCase())
    }

    List<SwaggerOperation> getOperations() {
        List<SwaggerOperation> result = []
        (spec.paths as Map<String,Map>)?.each { path, operations ->
            (operations as Map<String,Map>).each { operationType, operationSpec ->
                if (OPERATION_TYPES.contains(operationType)) {
                    result.add(new SwaggerOperation(
                            path: getPath(path),
                            operationType: operationType,
                            operationId: operationSpec.operationId as String,
                            controller: getController(path, operationSpec, operations[CONTROLLER_TAG] as String),
                            spec: operationSpec
                    ))
                } else if (!NO_OPERATION_PATH_ELEMENTS.contains(operationType)) {
                    log "Ignore operation type: $operationType"
                }
            }
        }
        result
    }

    String getPath(String path) {
        def basePath = spec.basePath as String
        if (basePath) {
            if (basePath.endsWith('/') && path.startsWith('/')) {
                basePath + path.substring(1)
            } else {
                basePath + path
            }
        } else {
            path
        }
    }

    String getController(String path, Map operationSpec, String controllerTag) {
        String result = getResourceCodeByPath(path) ?:
                controllerTag ?:
                        getResourceCodeByTag(operationSpec)
        assert result, "Can't determine controller name for $operationSpec.operationId"
        result
    }

    String getResourceCodeByPath(String path) {
        if (config?.resourceResolvers) {
            def resourcesCode = config.resourceResolvers.find { it.value.test(path)}?.key
            if (resourcesCode) {
                return resourcesCode
            }
            log "WARNING. Can't match path '$path' with config.resourceResolvers"
        }
        null
    }

    String getResourceCodeByTag(Map operationSpec) {
        List tags = (List) operationSpec.tags
        if (tags && tags.size() == 1) {
            return tags[0]
        }
        if (!tags) {
            log "WARNING. No tags specified for operation $operationSpec.operationId"
        } else if (tags.size() > 1) {
            log "WARNING. Too many tags for operation $operationSpec.operationId: $tags"
        }
        null
    }

    DataType getOperationRequestBody(Map spec) {
        Map bodySpec = (spec[PARAMETERS] as List<Map>)?.find { it.in == 'body'}
        if (!bodySpec) return null
        definitionsParser.toDataType(bodySpec.schema as Map)
    }

    List<Parameter> getOperationParameters(Map spec) {
        if (!spec[PARAMETERS]) {
            return []
        }
        List<Parameter> result = (spec[PARAMETERS] as List<Map>).findAll { it.in != 'body'}.collect { paramSpec ->
            if (paramSpec.containsKey(REF)) {
                builder.paramRef(parametersParser.getNameFromParamRef(paramSpec))
            } else {
                parametersParser.toParameter((Map) paramSpec)
            }
        }
        result
    }

    static List<String> getOperationConsumes(SwaggerOperation operation) {
        if (operation.spec.consumes) {
            return (List<String>) operation.spec.consumes
        }
        OperationType type = OperationType.valueOf(operation.operationType.toUpperCase())
        if (RestApiMetaModelChecker.isOperationWithBody(type)) {
            return ['application/json']
        }
        null
    }

    static List<String> getOperationProduces(SwaggerOperation operation) {
        (List<String>) operation.spec.produces ?: ['application/json']
    }

    List<HttpResponse> getOperationResponses(Map spec) {
        def result = []
        if (spec.responses == null) {
            throw new RuntimeException("No responses defined for operation: $spec.operationId")
        }
        (spec.responses as Map<String,Map>).each { status, desc ->
            def response = new HttpResponse(
                    status: 'default' == status ? 0 : Integer.valueOf(status),
                    description: desc.description as String,
            )
            if (desc.content) {
                // TODO
                throw new RuntimeException("TODO")
            } else {
                if (desc.schema) {
                    response.schema = definitionsParser.toDataSchema(desc.schema as Map)
                }
                result.add(response)
            }
        }
        result
    }
}
