/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.metamodel.parser.swagger

import pl.metaprogramming.metamodel.model.data.*

import static pl.metaprogramming.metamodel.model.data.DataType.*

class DefinitionsParser extends BaseParser {

    static final String FORMAT_DATE_TIME = 'date-time'
    static final String FORMAT_DATE = 'date'
    static final String MODEL_DEFS = '#/definitions/'
    static Set<String> PREDEFINED_FORMATS = [
            "string:$FORMAT_DATE",
            "string:$FORMAT_DATE_TIME",
            'integer:int32',
            'integer:int64',
            'number:float',
            'number:double',
    ] as Set<String>

    private Map<String, Closure<DataType>> DATA_TYPE_CREATORS = [
            'string': { Map spec ->
                if (spec.enum) {
                    enumParser.toEnumType(spec)
                } else if (spec.format == FORMAT_DATE || spec.format == FORMAT_DATE_TIME) {
                    DATE_TIME
                } else {
                    TEXT
                }
            },
            'integer' : { Map spec -> spec.format == 'int64' ? INT64 : INT32},
            'number' : { Map spec -> spec.format == 'float' ? FLOAT : spec.format == 'double' ? DOUBLE : DECIMAL },
            'boolean' : { BOOLEAN },
            'array' : { Map spec -> new ArrayType(itemsSchema: toDataSchema((Map) spec.items)) },
            'object' : { Map spec ->
                if (spec.additionalProperties) {
                    new MapType(valuesSchema: toDataSchema((Map) spec.additionalProperties))
                } else {
                    toObjectType(spec)
                }
            },
            'file' : { BINARY },
            'oauth2' : { TEXT },
            'apiKey' : { TEXT }
    ]

    EnumParser enumParser

    DefinitionsParser(BaseParser template, EnumParser enumParser) {
        configure(template)
        this.enumParser = enumParser
    }

    void readDefinitions() {
        log "Going to parse definitions..."
        Map<String, Map> definitions = new HashMap<>(spec.definitions as Map<String,Map>)
        if (spec.securityDefinitions) {
            definitions.putAll((spec.securityDefinitions as Map<String,Map>))
        }
        if (definitions) {
            def defNames = getOrderedDefinitions(definitions)
            defNames.each { code ->
                log "  $code"
                def schema = toRootDataSchema(definitions[code], code)
                if (schema.isObject()) {
                    schema.objectType.code = code
                    schema.objectType.description = schema.description
                }
                log "    $schema"
                builder.dataDef(code, schema)
            }
        }
    }

    List<String> getOrderedDefinitions(Map<String, Map> definitions) {
        List<String> result = []
        Map<String, Map> defsToOrder = [:]
        defsToOrder.putAll(definitions)
        int prevDefsToOrderSize = 0
        while (defsToOrder && defsToOrder.size() != prevDefsToOrderSize) {
            def defs = findDefinitionsWithAllowedDependencies(defsToOrder, result)
            result.addAll(defs)
            prevDefsToOrderSize = defsToOrder.size()
            defsToOrder.removeAll { defs.contains(it.key) }
        }
        if (defsToOrder) {
            throw new RuntimeException("Can't process definitions: ${defsToOrder.keySet()}")
        }
        result
    }

    List<String> findDefinitionsWithAllowedDependencies(Map<String, Map> definitions, List<String> allowedDependencies) {
        definitions.findAll { !hasUnknownDependencies(it.value, allowedDependencies)}.collect { it.key }
    }

    boolean hasUnknownDependencies(Map schemaSpec, List<String> knownSchemas) {

        if (schemaSpec[PROPERTIES]) {
            (schemaSpec[PROPERTIES] as Map<String,Map>).values().any { propSpec ->
                hasUnknownDependencies((Map)propSpec, knownSchemas)
            }
        } else if (schemaSpec[REF]) {
            def refSchema = getNameFromDefinitionRef(schemaSpec)
            !knownSchemas.contains(refSchema) && !externalSchemas.contains(refSchema)
        } else if (schemaSpec.items) {
            hasUnknownDependencies((Map) schemaSpec.items, knownSchemas)
        } else if (schemaSpec.allOf) {
            schemaSpec.allOf.any { Map childSpec ->
                hasUnknownDependencies(childSpec, knownSchemas)
            }
        } else {
            false
        }
    }

    ObjectType toObjectType(Map spec) {
        List<DataSchema> properties = []
        List<ObjectType> inherits = []
        if (spec.allOf) {
            spec.allOf.each { Map childSpec ->
                if (childSpec[REF]) {
                    inherits.add(builder.dataRef(getNameFromDefinitionRef(childSpec)).objectType)
                } else {
                    def childType = toObjectType(childSpec)
                    properties.addAll(childType.fields)
                    inherits.addAll(childType.inherits)
                }
            }
        } else {
            check(spec[PROPERTIES] != null && !(spec[PROPERTIES] as Map).isEmpty(),
                    'object should have properties', 'object should have properties')
            spec[PROPERTIES]?.each { Map.Entry<String, Map<String,String>> it ->
                properties.add(toDataSchema(it.value, it.key))
            }
            spec.required.each { propName ->
                def property = properties.find { it.code == propName}
                if (property) {
                    property.isRequired = true
                } else {
                    log "FAILED: Can't set required flag on $propName in object spec: $spec"
                }
            }
        }
        new ObjectType(fields: properties, inherits: inherits)
    }

    public <T extends DataSchema> T fillDataSchema(T schema, Map<String,String> spec) {
        schema.dataType = toDataType(spec)
        assert schema.dataType, "Can't handle $spec"
        schema.description = spec.description
        schema.format = getFormat(spec)
        schema.pattern = spec.pattern
        schema.minLength = toInt(spec.minLength)
        schema.maxLength = toInt(spec.maxLength)
        schema.minimum = spec.minimum
        schema.maximum = spec.maximum
        if (schema.isArray()) {
            schema.arrayType.minItems = toInt(spec.minItems)
            schema.arrayType.maxItems = toInt(spec.maxItems)
            shareAttributesWithItemSchema(schema)
        }
        schema
    }

    private void shareAttributesWithItemSchema(DataSchema schema) {
        if (schema.isArray()) {
            schema.arrayType.itemsSchema.attributes = schema.attributes
            shareAttributesWithItemSchema(schema.arrayType.itemsSchema)
        }
    }

    DataSchema toDataSchema(Map<String,String> spec, String code = null) {
        if (!code) {
            code = spec[REF] ? getNameFromDefinitionRef(spec) : null
        }
        fillDataSchema(new DataSchema(code: code ?: spec[REF] ? getNameFromDefinitionRef(spec) : null), spec)
    }

    DataSchema toRootDataSchema(Map<String,String> spec, String code) {
        def dataType = spec.enum ? enumParser.parse(spec, code) : toDataType(spec)
        new DataSchema(
                code: code,
                description: spec.description,
                dataType: dataType
        )
    }

    static String getFormat(Map spec) {
        switch (spec.format) {
            case FORMAT_DATE:
                return 'yyyy-MM-dd'
            case FORMAT_DATE_TIME:
                return "ISO8601"
            default:
                if (spec.format && !PREDEFINED_FORMATS.contains("${spec.type}:${spec.format}".toString())) {
                    println spec
                    return spec.format
                }
                return null
        }
    }

    DataType toDataType(Map spec) {
        def type = spec.type ?: spec[PROPERTIES] ? 'object' : null
        if (type) {
            if (!DATA_TYPE_CREATORS.containsKey(type)) {
                throw new RuntimeException("Can't handle data schema: $spec")
            }
            try {
                DATA_TYPE_CREATORS.get(type).call(spec)
            } catch (Exception e) {
                throw new RuntimeException("Can't handle data schema: $spec", e)
            }
        } else if (spec[REF]) {
            builder.dataRef(getNameFromDefinitionRef(spec)).dataType
        } else if (spec.allOf) {
            toObjectType(spec)
        } else {
            throw new RuntimeException("Can't handle data schema: $spec")
        }

    }

    static String getNameFromDefinitionRef(Map spec) {
        def ref = (String) spec[REF]
        ref.substring(ref.indexOf(MODEL_DEFS) + MODEL_DEFS.length())
    }

    static Integer toInt(Object value) {
        if (value instanceof Integer) return value
        if (value instanceof  String ) return new Integer(value)
        if (value != null) throw new RuntimeException("Can't transform to int: $value")
        null
    }
}
