package pl.metaprogramming.metamodel.builder

import pl.metaprogramming.metamodel.RestApiMetaModelChecker
import pl.metaprogramming.metamodel.model.data.DataSchema
import pl.metaprogramming.metamodel.model.data.DataType
import pl.metaprogramming.metamodel.model.data.ObjectType
import pl.metaprogramming.metamodel.model.rest.HttpResponse
import pl.metaprogramming.metamodel.model.rest.Operation
import pl.metaprogramming.metamodel.model.rest.Parameter
import pl.metaprogramming.metamodel.model.rest.RestApi
import pl.metaprogramming.metamodel.model.rest.enums.OperationType
import pl.metaprogramming.metamodel.model.rest.enums.ParamLocation

import java.util.function.Consumer

class RestApiBuilder {

    private static final String SUB_SCHEMA = 'Schema'
    private static final String SUB_PARAM = 'Parameters'
    private static final String SUB_RESP = 'Responses'

    RestApi model

    RestApiMetaModelChecker checker

    RestApiBuilder() {
        model = new RestApi()
        checker = new RestApiMetaModelChecker()
    }

    void dependsOn(List<RestApi> dependsOn) {
        this.model.dependsOn = dependsOn
    }


    /*
    void operation(Closure closure) {
        def o = Utils.apply(closure, new Operation())
        println "Operation: $o"
    }
    */

    void operation(String controller, String operationId, String path, OperationType operationType, String description, Consumer<Operation> configurator) {
        String fixedGroup = fixToJavaName(controller)
        String fixedOperationCode = fixToJavaName(operationId).uncapitalize()
        Operation operation = new Operation(
                group: fixedGroup.uncapitalize(),
                code: fixedOperationCode,
                path: path,
                type: operationType,
                description: description,
                consumes: model.consumes,
                produces: model.produces)
        configurator.accept(operation)
        checker.checkOperation(operation)
        model.operations.putIfAbsent(fixedGroup, [:])
        model.operations.get(fixedGroup).put(fixedOperationCode, operation)
    }

    private static String fixToJavaName(String value) {
        value.replaceAll(' ', '')
    }


    DataSchema dataDef(String code, String description, Map<String, DataSchema> properties) {
        dataDef(code, description, properties.collect {
            it.value.code = it.key
            it.value
        })
    }

    DataSchema dataDef(String code, String description, List<DataSchema> properties) {
        dataDef(code, new DataSchema(dataType: new ObjectType(fields: properties, description: description), description: description))
    }

    DataSchema dataDef(String code, DataSchema data) {
        checker.checkDataCode(code)
        data.code = code
        model.schemas.put(code, data)
    }



    void paramDef(String name, ParamLocation location, DataType schema, boolean isRequired, String description) {
        paramDef(name, new Parameter(name: name, location: location, dataType: schema, isRequired: isRequired, description: description))
    }

    void paramDef(String code, Parameter param) {
        checker.checkCommonParamCode(code)
        model.parameters.put(code, param)
    }

    List<Parameter> params(List paramList) {
        List<Parameter> result = []
        paramList.each {
            if (it instanceof String) {
                result.addAll(paramRef(it))
            } else if (it instanceof List) {
                if (it.size() == 2) {
                    def schema = (DataSchema) dataRef((String) it.get(0))
                    def location = (ParamLocation) it.get(1)
                    ((ObjectType) schema.dataType).fields.each {
                        result.add(new Parameter(
                                name: it.code,
                                location: location,
                                dataType: it.dataType,
                                isRequired: it.isRequired,
                                description: it.description,
                                format: it.format,
                        ))
                    }
                } else if (it.size() == 5) {
                    result.add(new Parameter(
                            name: (String) it.get(0),
                            location: (ParamLocation) it.get(1),
                            dataType: (DataType) it.get(2),
                            isRequired: (Boolean) it.get(3),
                            description: (String) it.get(4)))
                } else {
                    throw new RuntimeException("Can't define param: $it")
                }
            }
        }
        result
    }



    void responseDef(int code, String schemaRef, String description, String contentType = null) {
        checker.checkCommonResponseCode(code)
        model.responses.put(code, response(code, schemaRef, description, contentType))
    }

    HttpResponse response(int code, String schemaRef, String description, String contentType = null) {
        new HttpResponse(status: code, schema: dataRef(schemaRef), description: description, contentType: contentType)
    }

    List<HttpResponse> responses(List responseList) {
        (List<HttpResponse>) responseList.collect {
            it instanceof HttpResponse ? it : responseRef((Integer) it)
        }
    }


    DataSchema dataRef(String code) {
        (DataSchema) getRequired(SUB_SCHEMA, code)
    }

    HttpResponse responseRef(Integer responseCode) {
        (HttpResponse) getRequired(SUB_RESP, responseCode)
    }

    Parameter paramRef(String code) {
        (Parameter) getRequired(SUB_PARAM, code)
    }

    private def getRequired(String subject, def key) {
        def result = getValue(model, subject, key)
        assert result, "$subject '$key' not defined"
        result
    }

    static def getValue(RestApi model, String subject, def key) {
        Map map = subject == SUB_SCHEMA ? model.schemas
                : subject == SUB_PARAM ? model.parameters
                : subject == SUB_RESP ? model.responses
                : null
        map.containsKey(key) ? map.get(key) : model.dependsOn.findResult { getValue(it, subject, key) }
    }

}
